// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'create_playlist_button.dart';
import 'database_info_widget.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'screen.dart';
import 'section_heading_widget.dart';
import 'table_model.dart';
import 'table_widget.dart';
import 'tables_model.dart';

/// Display a database.
///
/// This widget fills a tab with lots of [TableWidget]s, one for each table in the database.
class DatabaseViewerWidget extends StatelessWidget {
  final String dbPath;
  final TablesModel tablesNotifier;

  const DatabaseViewerWidget({super.key, required this.dbPath, required this.tablesNotifier});

  @override
  Widget build(BuildContext context) {
    AppState.debug('DatabaseViewerWidget::build()');

    bool screenIsCompact = Screen.isCompact(context);
    Map tableLayout = screenIsCompact ? BS.tableLayoutCompact : BS.tableLayoutStandard;
    double sidePadding = screenIsCompact ? BS.paddingMedium : BS.paddingMedium;

    ThemeData themeData = DbManager.getDbCurrentThemeData(dbPath);
    ColorScheme colorScheme = themeData.colorScheme;

    Color mainColor = Color.alphaBlend(
      colorScheme.secondary.withValues(alpha: 0.2),
      colorScheme.onSurface,
    );

    Color topBackgroundColor = Color.alphaBlend(
      colorScheme.secondaryContainer.withValues(alpha: 0.5),
      colorScheme.surface,
    );

    /// Set as many default colours/styles as possible here, relatively high up in the widget tree
    themeData = themeData.copyWith(
      textTheme: themeData.textTheme.apply(bodyColor: mainColor, displayColor: mainColor),
      iconTheme: themeData.iconTheme.copyWith(color: mainColor),
      dataTableTheme: themeData.dataTableTheme.copyWith(
        headingRowColor: WidgetStatePropertyAll(colorScheme.surface),
        headingRowHeight: tableLayout['headingRowHeight'],
        headingTextStyle: tableLayout['headingTextStyle'],
        checkboxHorizontalMargin: tableLayout['checkboxHorizontalMargin'],
        dividerThickness: 0,
        columnSpacing: tableLayout['columnSpacing'],
        horizontalMargin: tableLayout['horizontalMargin'],
      ),
      scrollbarTheme: themeData.scrollbarTheme.copyWith(
        thickness: WidgetStatePropertyAll(BS.scrollbarThicknessThin),
        thumbColor: WidgetStatePropertyAll(colorScheme.secondary),
      ),
      textSelectionTheme: themeData.textSelectionTheme.copyWith(
        selectionColor: Color.alphaBlend(
          colorScheme.tertiaryContainer.withValues(alpha: 0.7),
          colorScheme.surface,
        ),
      ),
      checkboxTheme: themeData.checkboxTheme.copyWith(
        side: BorderSide(color: colorScheme.primary, width: 2),
      ),
    );

    TableModel bookmarkedPlaylistsTableNotifier = tablesNotifier.bookmarkedPlaylistsNotifier;
    TableModel channelSubscriptionsTableNotifier = tablesNotifier.channelSubscriptionsNotifier;

    List<Widget> items = [
      /// Optional database info
      if (AppState.getPreference('dbInfoToShow') != DbInfoToShow.none.name)
        DatabaseInfoWidget(dbPath: dbPath, backgroundColor: topBackgroundColor),
      //
      SizedBox(height: BS.paddingExtraLarge + BS.paddingMedium),

      ///
      /// Bookmarked Playlists and Channel Subs
      ///
      Padding(
        padding: const EdgeInsets.only(bottom: BS.paddingLarge),
        child: ListenableBuilder(
          listenable: bookmarkedPlaylistsTableNotifier,
          builder: (context, child) {
            return TableWidget(
              tableNotifier: bookmarkedPlaylistsTableNotifier,
              dbPath: dbPath,
              key: ValueKey(bookmarkedPlaylistsTableNotifier.tablePath),
            );
          },
        ),
      ),
      Padding(
        padding: const EdgeInsets.only(bottom: BS.paddingLarge),
        child: ListenableBuilder(
          listenable: channelSubscriptionsTableNotifier,
          builder: (context, child) {
            return TableWidget(
              tableNotifier: channelSubscriptionsTableNotifier,
              dbPath: dbPath,
              key: ValueKey(channelSubscriptionsTableNotifier.tablePath),
            );
          },
        ),
      ),

      ///
      /// Search (special temporary local playlist)
      ///
      SectionHeadingWidget('Search'),

      ...tablesNotifier.searchPlaylists.entries.map<Widget>(
        (MapEntry<int, TableModel> tableEntry) => Padding(
          padding: const EdgeInsets.only(bottom: BS.paddingLarge),
          child: ListenableBuilder(
            listenable: tableEntry.value,
            builder: (context, child) {
              return TableWidget(
                tableNotifier: tableEntry.value,
                dbPath: dbPath,
                key: ValueKey(tableEntry.value.tablePath),
              );
            },
          ),
        ),
      ),

      ///
      /// Custom playlists
      ///
      SectionHeadingWidget('Custom Playlists'),

      ...tablesNotifier.localPlaylists.entries.map<Widget>(
        (MapEntry<int, TableModel> tableEntry) => Padding(
          padding: const EdgeInsets.only(bottom: BS.paddingLarge),
          child: ListenableBuilder(
            listenable: tableEntry.value,
            builder: (context, child) {
              return TableWidget(
                tableNotifier: tableEntry.value,
                dbPath: dbPath,
                key: ValueKey(tableEntry.value.tablePath),
              );
            },
          ),
        ),
      ),

      ///
      /// 'Create new' button
      ///
      CreatePlaylistButton(dbPath: dbPath),

      const SizedBox(height: BS.paddingExtraLarge),
    ];

    return Theme(
      data: themeData,
      child: Builder(
        builder: (context) {
          /// [Text] doesn't inherit styles from above so we need to wrap everything in
          /// [DefaultTextStyle].
          return DefaultTextStyle.merge(
            style: TextStyle(color: mainColor),
            child: Container(
              decoration: BoxDecoration(
                color: colorScheme.surface,
                border: Border(
                  top: BorderSide(color: topBackgroundColor, width: BS.accentBorderThickness),
                ),
              ),
              child: SingleChildScrollView(
                child: Padding(
                  padding: EdgeInsets.only(bottom: BS.paddingMedium, right: sidePadding),
                  child: Column(crossAxisAlignment: CrossAxisAlignment.start, children: items),
                ),
              ),
            ),
          );
        },
      ),
    );
  }
}
