// SPDX-License-Identifier: GPL-3.0-only

import 'dart:async';
import 'package:flutter/foundation.dart';

import 'package:flutter/material.dart';
import 'package:observable/observable.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'constants.dart';

/// Add this widget to the tree to enable shared state in the app.
///
/// All widgets which want to access shared state should be children of this
/// widget. Make this widget high enough in the tree that it wraps all of the
/// children which need it, but no higher (for efficiency).
class AppStateWidget extends StatelessWidget {
  final Widget child;

  /// Set some default empty variables.
  static final ObservableMap _appState = ObservableMap.from({
    'preferences': null,
    'debug': '',
    'totalDbsOpenedSinceSessionStart': 0,
    'fileLoadOrSaveIsInProgress': false,
    'dialogToShow': {},
    'databases': {},
    'currentSelectedDbPath': '',
    'aDatabaseWasDirtied': null,
  });

  const AppStateWidget({super.key, required this.child});

  @override
  Widget build(BuildContext context) {
    return child;
  }
}

/// Contain and manage shared app state.
///
/// Any widget that wants to access app state should extend this class in its
/// [createState], eg:
/// ``` dart
/// class _TabManagerWidgetState extends AppState<TabManagerWidget>
/// ```
/// When a change is made to any element in [listenForChanges], the widget will
/// be rebuilt.
abstract class AppState<T extends StatefulWidget> extends State<T> {
  StreamSubscription? _appStateChangeSubscription;

  /// List of [_appState] keys on which extending classes will listen for
  /// changes. When the value related to a listened key changes, this widget
  /// will rebuild, eg:
  ///
  /// ``` dart
  /// listenForChanges = ['mainData', 'userNames'];
  /// ```
  /// The above code means this widget will be rebuilt when
  /// [_appState['mainData'] or [_appState['userNames']] is changed.
  List<String>? get listenForChanges;
  set listenForChanges(List<String>? value);

  @override
  void initState() {
    super.initState();

    /// Listen for changes in [_appState] and rebuild any interested widgets
    /// (those which are listening for the specific key which changed).
    _appStateChangeSubscription = AppStateWidget._appState.changes.listen((List event) {
      for (ChangeRecord change in event) {
        if (change is MapChangeRecord) {
          if (listenForChanges != null && listenForChanges!.contains(change.key)) {
            String widgetDescription =
                widget.key?.toString() ?? '${widget.toStringShort()}::${widget.hashCode}';

            /// if the key was removed, that indicates this is just the first
            /// step in a forced rebuild, so ignore this change - another
            /// change of the same key will follow immediately and we don't
            /// want to rebuild twice.
            if (!change.isRemove) {
              //AppState.debug('$widgetDescription HEARD: \'${change.key}\'');
              setState(() {});
            }
          }
        }
      }
    });
  }

  @override
  void dispose() {
    super.dispose();
    _appStateChangeSubscription?.cancel();
  }

  /// Update a part (based on [key]) of the shared state of the app.
  static void update(String key, dynamic newValue, {forceRebuild = false}) {
    if (forceRebuild) {
      /// Sometimes [newValue] may be an already-existing object, where only
      /// the internals have changed. However, because the object is 'still the
      /// same object' (as it is only looked at in a shallow way),
      /// `AppStateWidget._appState.changes()` won't fire.
      ///
      /// In this case, `forceRebuild` can be passed as `true` to work around
      /// this non-detection of the internal object changes. If so, we remove
      /// the object, then add it again, forcing the change to be recognised.
      ///
      /// This causes the `changes()` event to fire twice, first when the object
      /// is removed, then again when the object is re-added. So
      /// `_appStateChangeSubscription` has to watch out for this and ignore
      /// the first change (otherwise the widget tree gets rebuilt twice).
      AppStateWidget._appState.remove(key);
      //AppState.debug('Force REBUILD for widgets listening to: \'${key}\'');
    }
    AppStateWidget._appState[key] = newValue;
  }

  /// Get a part (based on [key]) of the shared state of the app.
  static dynamic get(String key) {
    return AppStateWidget._appState[key];
  }

  /// Output debugging info.
  ///
  /// Print the info to the console, and if `alwaysOnScreen` is `true`, also
  /// display the info in a dialog.
  static void debug(str, {alwaysOnScreen = false}) {
    if (kDebugMode) {
      print(str);
    }
    if (alwaysOnScreen) {
      AppState.update('debug', str.toString());
      //AppState.update('debug', '${AppState.get('debug')}\n${str.toString()}');
      showAppDialog(title: 'Error', message: AppState.get('debug'), isError: true);
    }
  }

  /// Pass debugging info to the app's main dialog for display.
  static void showAppDialog({
    required String title,
    String? message,
    Widget? content,
    List<Widget>? actions,
    bool isError = false,
  }) {
    AppState.update('dialogToShow', {
      'title': title,
      'content': content,
      'message': message,
      'isError': isError,
      'actions': actions,
    }, forceRebuild: true);
  }

  /// Get the current route/page we are on
  static BuildContext get globalContext {
    return AppState.get('mainNavigatorKey').currentContext;
  }

  /// [SharedPreferencesWithCache] must be use the [SharedPreferencesWithCache.create] constructor,
  /// which is async.
  /// https://pub.dev/packages/shared_preferences
  static Future<void> initPreferences() async {
    AppState.update(
      'preferences',
      await SharedPreferencesWithCache.create(
        cacheOptions: const SharedPreferencesWithCacheOptions(),
      ),
    );
  }

  /// Sometimes we want to trigger a rebuild for certain widgets without a specific variable
  /// changing, or we want to bundle lots of changes into one concept, eg when any of preferences
  /// for [useCoverImages], [decorationLevelIndex], [themeBrightnessModeIndex] have changed,
  /// it's nicer for widgets to just be able to listen for [themeChange]. These 'states' just hold
  /// nulls as we only need the key, not the value.
  static void forceRebuildOn(String key) {
    AppState.update(key, AppState.get(key), forceRebuild: true);
  }

  static Future<void> setPreference(String key, dynamic value, {forceRebuild = false}) async {
    SharedPreferencesWithCache? sharedPreferences = AppState.get('preferences');

    if (sharedPreferences == null) {
      AppState.debug('[SharedPreferencesWithCache] has not yet been initialised');
    } else if (!BS.defaultPreferences.containsKey(key)) {
      AppState.debug(
        "Preference '$key' does not exist in app (not found in 'BS.defaultPreferences')",
      );
    } else {
      var [defaultType, defaultValue] = BS.defaultPreferences[key];
      if (value.runtimeType != defaultType) {
        AppState.debug(
          "Trying to set preference to incorrect type. '$key' should be '$defaultType', not '${value.runtimeType}'",
        );
      } else {
        switch (defaultType) {
          case const (bool):
            await sharedPreferences.setBool(key, value);
          case const (String):
            await sharedPreferences.setString(key, value);
          case const (double):
            await sharedPreferences.setDouble(key, value);
          case const (int):
            await sharedPreferences.setInt(key, value);
        }
      }
    }
    //AppState.debug("SET $key: ${value.toString()}");

    AppState.update('preferences', sharedPreferences, forceRebuild: forceRebuild);
  }

  static dynamic getPreference(String key) {
    SharedPreferencesWithCache? sharedPreferences = AppState.get('preferences');
    dynamic userPreference;

    if (sharedPreferences == null) {
      AppState.debug('SharedPreferencesWithCache has not yet been initialised: $key');
      if (BS.defaultPreferences.containsKey(key)) {
        var [defaultType, defaultValue] = BS.defaultPreferences[key];
        userPreference = defaultValue;
        AppState.debug('- returning default ($defaultValue)');
      }
    } else if (BS.defaultPreferences.containsKey(key)) {
      var [defaultType, defaultValue] = BS.defaultPreferences[key];
      switch (defaultType) {
        case const (bool):
          userPreference = sharedPreferences.getBool(key) ?? defaultValue;
        case const (String):
          userPreference = sharedPreferences.getString(key) ?? defaultValue;
        case const (double):
          userPreference = sharedPreferences.getDouble(key) ?? defaultValue;
        case const (int):
          userPreference = sharedPreferences.getInt(key) ?? defaultValue;
      }
    } else {
      AppState.debug(
        "Shared preference '$key' does not exist in app (not found in 'BS.defaultPreferences')",
      );
    }
    //AppState.debug("GET $key: ${userPreference.toString()}");

    return userPreference;
  }

  /// Debounce a function. [voidFunction] must not return anything. [callerKey] is used to form
  /// a [Timer] variable name, so should be something unique otherwise different functions might in
  /// theory clash.
  static void debounceVoidFunction({
    required String callerKey,
    required Function voidFunction,
    Duration debounceDuration = BS.defaultDebounceDuration,
  }) {
    String timerId = callerKey + voidFunction.hashCode.toString();
    AppState.debug('timerId: $timerId');

    /// Cancel any existing timers for the same function (this is where our debounce happens).
    Timer? existingTimer = AppState.get(timerId);
    if (existingTimer != null) {
      existingTimer.cancel();
    }

    /// Save the timer in [AppState]
    AppState.update(
      timerId,
      Timer(debounceDuration, () {
        voidFunction.call();

        /// After calling the debounced function, clean up the [Timer].
        AppStateWidget._appState.remove(timerId);
      }),
    );
  }

  static bool get isDesktop =>
      defaultTargetPlatform == TargetPlatform.windows ||
      defaultTargetPlatform == TargetPlatform.linux ||
      defaultTargetPlatform == TargetPlatform.macOS;
}
