extends Line2D

class_name SnakeBase

signal eat(who: SnakeBase, type: bool)
signal die(who: SnakeBase, positions: Array[Vector2])

# snake line2d initial config
const initial_num_points: int = 39
const segment_length: int = 3
const names_file = "res://data/names.txt"

const dir_speed_factor: float = 4.

@onready var num_points: int = initial_num_points
@onready var snake_length: float = (num_points-1) * segment_length
@onready var current_length: float = snake_length
@onready var snake_width: float = log(snake_length)*3
@onready var snake_head: Area2D = $Head
@onready var snake_body: Area2D = $Body

# movement method
var movement

# initial direction (angle in radians) and speed
var direction: float
var speed: float

var max_rotation: float

var start_speed: float

var snake_name: String

# private variables
var _snake_names = GetResources.snake_names
var _heads = GetResources.heads
var _textures = GetResources.textures
var _noises = GetResources.noises

# the collision shapes for the snake body
var circle: CircleShape2D = CircleShape2D.new()

func _get_index(orig: int, maximum: int) -> int:
	"""Return a probably random index in range 0-maximum."""
	if orig == clamp(orig, 0, maximum-1):
		return orig
	else:
		return randi()%maximum

func _get_color(color: String) -> Color:
	"""Return a given or a random Color."""
	if color:
		return Color(color)
	else:
		return Color(randf(), randf(), randf())

func get_class_name() -> String:
	return "SnakeBase"

func initialize(move: int = 0, show_label: bool = true, \
				my_name: String = "", head_nr: int = -1, body_nr: int = -1, \
				noise_nr: int = -1, pattern_color: String = "", fill_color: String = "") -> void:
	circle.radius = 1.5

	# set movement mode
	movement = clamp(move, 0, 1)
	
	if my_name:
		snake_name = my_name
	else:
		snake_name = _snake_names[randi()%_snake_names.size()]

	if show_label:
		$NameLabel.text = snake_name
	else:
		$NameLabel.hide()

	# body texture
	texture = _textures[_get_index(body_nr, _textures.size())]
	
	# set material parameters
	var mat: Material = material.duplicate()
	mat.set_shader_parameter("noise_pattern", _noises[_get_index(noise_nr, _noises.size())])
	mat.set_shader_parameter("selected_pattern", _get_color(pattern_color))
	mat.set_shader_parameter("selected_fill", _get_color(fill_color))
	material = mat

	# set head pattern
	$Head/head_pattern.texture = _heads[_get_index(head_nr, _heads.size())]
	$Head/head_pattern.material = mat

	$Head.scale = snake_width/30. * Vector2.ONE

# Called every frame. 'delta' is the elapsed time since the previous frame.
func _process(_delta: float) -> void:
	self.width = snake_width
	_move(_delta)

func _move(speed_factor: float) -> void:
	"""Move the snake."""
	var new_position: Vector2
	var point_count = get_point_count()
	var head_index = point_count - 1
	var dir: Vector2 = Vector2(cos(direction), sin(direction))
	var my_speed = 10 * speed * speed_factor
	var head_pos: Vector2 = get_point_position(head_index) + my_speed * dir

	if movement == 0:
		add_point(head_pos)
		current_length += my_speed
		while current_length > snake_length + segment_length/2.:
			var tail_length = (get_point_position(1)-get_point_position(0)).length()
			current_length -= tail_length
			remove_point(0)

	elif movement == 1:
		set_point_position(head_index, head_pos)
		for point_index in range(point_count-2, -1, -1):
			var current_point_pos: Vector2 = get_point_position(point_index)
			var prev_point_pos: Vector2 = get_point_position(point_index + 1)
			new_position = (current_point_pos - prev_point_pos).normalized() * segment_length + prev_point_pos
			set_point_position(point_index, new_position)

	# set position of head CollisionShape
	snake_head.position = head_pos
	snake_head.rotation = direction

	# set label position
	$NameLabel.position = head_pos + Vector2(0, -36)

	_update_body_collision_shape()

func _grow() -> void:
	"""Let the snake grow in length and width."""
	var grow_now = 0.9
	if randf() > grow_now:
		var dir = (get_point_position(0)-get_point_position(1)).normalized()
		add_point(get_point_position(0) + segment_length * dir, 0)
		num_points += 1
		snake_length = (num_points-1) * segment_length
		snake_width = log(snake_length) * 3
		snake_head.scale = snake_width/30. * Vector2.ONE

func get_length() -> float:
	"""Return the current length."""
	return snake_length

func _appear(start_position: Vector2):
	"""Let the snake appear at the specified position."""
	# clear all points
	for i in get_point_count():
		remove_point(0)
	# start in the middle of the playground
	var pos: Vector2
	var x = [0, -segment_length*sqrt(3.)/2., -segment_length*sqrt(3.)/2.]
	var y = [0, segment_length/2., -segment_length/2.]
	for i in num_points:
		pos = start_position + Vector2(x[i%3], y[i%3])
		add_point(pos, 0)

	# set head to center and activete head collision shape
	$Head.position = Vector2.ZERO
	$Head.monitoring = true

	set_process(true)
	show()

func _reset() -> void:
	"""Reset the length of the snake."""
	num_points = initial_num_points

func start(start_position: Vector2, start_direction: float, _start_speed: float, reset: bool = true) -> void:
	"""Start the snake's movement."""
	direction = start_direction
	start_speed = _start_speed
	speed = _start_speed
	max_rotation = deg_to_rad(dir_speed_factor*log(speed))

	if reset:
		_reset()
	_appear(start_position)

func _clamp_angle(angle: float, _min: float, _max: float) -> float:
	"""Clamp an angle."""
	var n_min = angle_difference(angle, _min)
	var n_max = angle_difference(angle, _max)
	if n_min <= 0 and n_max >= 0:
		return angle
	if abs(n_min) < abs(n_max):
		return _min
	return _max

func _add_collision_circle() -> void:
	"""Add a collision child of $Body."""
	var coll_shape = CollisionShape2D.new()
	coll_shape.shape = circle
	$Body.call_deferred("add_child", coll_shape)

func _update_body_collision_shape() -> void:
	"""Move the collision shapes along the snake body."""
	var n = get_point_count()
	# add collision shapes
	if $Body.get_child_count() < n:
		_add_collision_circle()
	# remove collision shapes if snake points have decreased
	if $Body.get_child_count() > n:
		$Body.get_child(0).free()
	for i in $Body.get_child_count():
		var cs = $Body.get_child(i)
		cs.position = get_point_position(clamp(i, 0, n-1))

func _die():
	snake_head.set_deferred("monitoring", false)
	
	set_process(false)
	for coll in $Body.get_children():
		coll.free()
	hide()
	# copy snake positions
	var positions: Array[Vector2] = []
	for i in int(get_point_count()/(4.*segment_length)):
		positions.append(get_point_position(4*segment_length*i))
	die.emit(self, positions)

	
func _on_head_body_entered(_body) -> void:
	if _body.is_in_group("food") or _body.is_in_group("superfood"):
		# eating some food
		_body.call_deferred("queue_free")
		_grow()
		eat.emit(self, _body.is_in_group("food"))
	else:
		_die()

func _on_head_area_entered(area: Area2D) -> void:
	# die only when touching someone else
	if area != $Body:
		_die()
