extends Node2D

# general settings
var background_image: int
var music_on: bool
var show_label: bool
var num_snakes: int
var food_pieces: int
var start_speed: float
var movement: int
var controls_position: int

@onready var general_settings = GlobalConfigFile.load_general_settings()
@onready var backgrounds = GetResources.backgrounds

@onready var high_scores
@onready var camera: Camera2D = $Camera
@onready var lengths_table: PanelContainer = $Camera/LengthsTable

# controls
@onready var pause: Node2D = $Camera/Pause
@onready var velocity_control: Node2D = $Camera/VelocityControl
@onready var joystick: Node2D = $Camera/Joystick

var my_name: String
var head_nr: int
var body_nr: int
var noise_nr: int
var pattern_color: String
var fill_color: String

var length_dict: Dictionary
var direction_dict: Dictionary

var my_rank: int = 0

var food_scene: PackedScene = preload("res://scenes/food.tscn")
var superfood_scene: PackedScene = preload("res://scenes/superfood.tscn")

var snake_scene: PackedScene = preload("res://scenes/computer_snake.tscn")

@onready var player = $PlayerSnake

var computer_snakes: Array

var score: float = 0
var score_inc: float = 1.
var minx: int = -1600
var miny: int = -1100
var maxx: int = 1600
var maxy: int = 1100

# Called when the node enters the scene tree for the first time.
func _ready():
	background_image = general_settings["background_image"]
	music_on = general_settings["music_on"]
	show_label = general_settings["show_names"]
	num_snakes = general_settings["snakes_number"]
	food_pieces = general_settings["food_pieces"]
	start_speed = general_settings["snake_speed"]
	movement = general_settings["snake_movement"]
	controls_position = general_settings["controls_position"]
	
	score_inc = start_speed / 10.
	
	var player_settings = GlobalConfigFile.load_player_settings(0)
	my_name = player_settings["name"]
	head_nr = player_settings["head"]
	body_nr = player_settings["body"]
	noise_nr = player_settings["noise"]
	pattern_color = player_settings["pattern_color"]
	fill_color = player_settings["fill_color"]
	
	player.initialize(
		movement, show_label, my_name, head_nr, body_nr, \
		noise_nr, pattern_color, fill_color)
	player.connect("eat", eating)
	player.connect("die", dying)
	# Set my_name to eventually random player name
	my_name = player.snake_name
	length_dict[my_name] = player.get_length()
	
	$Playground.texture = backgrounds[background_image]
	
	if music_on:
		$Music.play()

	# mirror controls
	camera.scale.x = controls_position * 2 - 1
	for child in camera.get_children():
		child.scale.x = controls_position * 2 - 1
	# if controls are on the left adjust position
	if controls_position == 0:
		lengths_table.position.x += lengths_table.size.x
		$Camera/Pause.position.x += 66
		$Camera/VelocityControl.position.x += 66

	high_scores = HighScores.load_or_create()

	new_game()
	$LengthTimer.start()
	$FoodTimer.start()
	#Input.set_mouse_mode(Input.MOUSE_MODE_HIDDEN)

func _process(_delta: float) -> void:
	if player.is_processing():
		# set camera to player's head
		camera.position = player.snake_head.position
	else:
		# move camera back to origin
		camera.position -= camera.position * _delta/2

func new_game():
	my_rank = num_snakes
	score = 0.
	remove_food()
	add_food()
	for i in num_snakes:
		add_computer_snake()
	
	player.start(Vector2.ZERO, 0, start_speed, true)
	$HUD.update_score(roundi(score))

func reanimate_computer_snake(who):
	# after a random time re-animate the computer snake
	var waiting = randf_range(5., 10.)
	await get_tree().create_timer(waiting, false).timeout
	var snake_position = Vector2(randf_range(minx+100, maxx-100), randf_range(miny+100, maxy-100))
	var snake_direction = randf_range(0, 360)
	$HUD.show_message(tr("PLAYER_REANIMATED")%who.snake_name)
	who.start(snake_position, snake_direction, start_speed, false)

func remove_food():
	var all_food = get_tree().get_nodes_in_group("food")
	all_food.append_array(get_tree().get_nodes_in_group("superfood"))
	for food in all_food:
		food.call_deferred("queue_free")
	
func add_food() -> void:
	var all_food = get_tree().get_nodes_in_group("food")
	var number = food_pieces - all_food.size()
	for i in number:
		var food = food_scene.instantiate()
		self.call_deferred("add_child", food)
		var food_location = Vector2(randf_range(minx+100, maxx-100), randf_range(miny+100, maxy-100))
		food.position = food_location
		food.add_to_group("food")

func eating(who, type):
	if who.get_class_name() == "PlayerSnake":
		if type:
			score += score_inc
			$EatSound.play()
		else:
			score += 5 * score_inc
			$EatSuperFoodSound.play()
		$HUD.update_score(roundi(score))

	# update length_dict
	var length = who.get_length()
	length_dict[who.snake_name] = length

func dying(who, positions):
	var tex = who.texture
	var mat = who.material
	for pos in positions:
		var superfood = superfood_scene.instantiate()
		self.call_deferred("add_child", superfood)
		superfood.init(tex, mat)
		superfood.add_to_group("superfood")
		superfood.position = pos

	if who.get_class_name() == "PlayerSnake":
		$PlayerDied.play()
		$HUD.show_player_died()
	else:
		$HUD.show_message(tr("PLAYER_DIED")%who.snake_name)
		reanimate_computer_snake(who)

func add_computer_snake() -> void:
	var snake = snake_scene.instantiate()
	snake.initialize(movement, show_label)
	self.call_deferred("add_child", snake)
	var snake_position = Vector2(randf_range(minx+100, maxx-100), randf_range(miny+100, maxy-100))
	var snake_direction = randf_range(0, 360)
	snake.connect("eat", eating)
	snake.connect("die", dying)
	snake.start(snake_position, snake_direction, start_speed)
	length_dict[snake.snake_name] = snake.get_length()
	direction_dict[snake.snake_name] = 0
	computer_snakes.append(snake)


func update_length():
	var keys: Array = length_dict.keys()
	var lengths: Array = length_dict.values()
	var length_list = Array()
	
	# Sort keys in descending order of values.
	keys.sort_custom(func(x, y): return length_dict[x] > length_dict[y])

	# get directions
	for snake in computer_snakes:
		var angle = (snake.snake_head.position - player.snake_head.position).angle()
		direction_dict[snake.snake_name] = angle
		
	var in_top: bool = false
	for i in keys.size():
		if i > 10 and in_top:
			break
		if keys[i] == my_name:
			in_top = true
			my_rank = i
			length_list.append(["%d."%(i+1), keys[i], "%.2f m " % (length_dict[keys[i]]/100.), true, 0])
		else:
			if i+1 < 10 + int(in_top):
				length_list.append(
					["%d."%(i+1), keys[i], "%.2f m " % (length_dict[keys[i]]/100.), false, direction_dict[keys[i]]])
	
	lengths_table.update_lengths(length_list)
	
	# if one snake reaches 10m length the game is over
	if lengths.max() >= 1000.:
		game_over(length_list[0][1], length_list[0][2])
		# get extra points for your rank
		# just wait a little bit before counting the bonus
		await get_tree().create_timer(2.).timeout
		# add 100 points per opponent behind
		for i in num_snakes-my_rank:
				await get_tree().create_timer(.1).timeout
				score += 100
				$EatSuperFoodSound.play()
				$HUD.update_score(roundi(score))

		# add another hundred points if player is first
		if my_rank == 0:
			score += 100
			
		await get_tree().create_timer(.1).timeout
		$HUD.update_score(roundi(score), true)

func control_visible(on: bool) -> void:
	joystick.visible = on
	velocity_control.visible = on
	pause.visible = on
	
func game_over(_name, _length) -> void:
	# pause execution and wait a short time
	get_tree().paused = true
	# play game over sound
	$GameOver.play()
	# remove contol elements
	control_visible(false)
	# show game over HUD
	$HUD.show_game_over(_name, _length)

func save_highscore() -> void:
	high_scores.add_entry([player.snake_name, score])
	high_scores.save()
	
func pause_game() -> void:
	get_tree().paused = pause.paused

func accelerate_player(amount: float) -> void:
	player.accelerate(amount)

func _on_length_timer_timeout() -> void:
	update_length()
	
func _on_food_timer_timeout() -> void:
	add_food()

func reanimate():
	player.start(Vector2.ZERO, 0, start_speed, false)
	
func _on_hud_restart_game() -> void:
	save_highscore()
	get_tree().reload_current_scene()

func _on_hud_quit_game() -> void:
	save_highscore()
	get_tree().change_scene_to_file("res://scenes/menu.tscn")
