extends Control

signal save

@onready var snake_panel: Panel = $MarginContainer/VBoxContainer/HBoxContainer/SnakePanel
@onready var snake_name_edit: LineEdit = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/SnakeNameEdit
@onready var snake_head_button: OptionButton = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/Textures/SnakeHeadButton
@onready var snake_body_button: OptionButton = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/Textures/SnakeBodyButton
@onready var snake_noise_button: OptionButton = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/Textures/SnakeNoiseButton
@onready var pattern_color_picker_button: ColorPickerButton = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/PatternColorPickerButton
@onready var fill_color_picker_button: ColorPickerButton = $MarginContainer/VBoxContainer/HBoxContainer/EditContainer/FillColorPickerButton

var selection: int

var head_index: int
var body_index: int
var noise_index: int

func _reduce_color_picker(cp: ColorPicker) -> void:
	cp.color_modes_visible = false
	cp.sampler_visible = false
	cp.sliders_visible = false
	cp.presets_visible = false
	cp.hex_visible = false

func _add_items(ob: OptionButton, icons: Array, index: int) -> void:
	for idx in icons.size():
		var img = icons[idx].get_image()
		img.resize(48, 48)
		var itex = ImageTexture.create_from_image(img)
		ob.add_icon_item(itex, "")
	ob.selected = index
	
func set_selection(nr: int) -> void:
	selection = nr
	snake_panel.initialize(selection)
	
	var snake_settings = GlobalConfigFile.load_player_settings(selection)
	snake_name_edit.text = snake_settings["name"]
	
	head_index = snake_settings["head"]
	body_index = snake_settings["body"]
	noise_index = snake_settings["noise"]

	snake_head_button.selected = head_index
	snake_body_button.selected = body_index
	snake_noise_button.selected = noise_index
	
	pattern_color_picker_button.color = snake_panel.mat.get_shader_parameter("selected_pattern")
	fill_color_picker_button.color = snake_panel.mat.get_shader_parameter("selected_fill")
	
func _ready():
	set_selection(0)
	_add_items(snake_head_button, snake_panel.heads, head_index)
	_add_items(snake_body_button, snake_panel.textures, body_index)
	_add_items(snake_noise_button, snake_panel.noises, noise_index)
	
	var pattern_color_picker = pattern_color_picker_button.get_picker()
	var fill_color_picker = fill_color_picker_button.get_picker()
	_reduce_color_picker(pattern_color_picker)
	_reduce_color_picker(fill_color_picker)
	
func _on_exit_button_pressed() -> void:
	# Save settings
	GlobalConfigFile.save_player_settings(selection, "name", snake_name_edit.text.capitalize())
	GlobalConfigFile.save_player_settings(selection, "head", head_index)
	GlobalConfigFile.save_player_settings(selection, "body", body_index)
	GlobalConfigFile.save_player_settings(selection, "noise", noise_index)
	
	GlobalConfigFile.save_player_settings(selection, "pattern_color", pattern_color_picker_button.color.to_html(false))
	GlobalConfigFile.save_player_settings(selection, "fill_color", fill_color_picker_button.color.to_html(false))
	
	save.emit()
	
func _on_snake_head_button_item_selected(index: int) -> void:
	head_index = index
	snake_panel.head.texture = snake_panel.heads[index]

func _on_snake_body_button_item_selected(index: int) -> void:
	body_index = index
	snake_panel.body.texture = snake_panel.textures[index]

func _on_snake_noise_button_item_selected(index: int) -> void:
	noise_index = index
	snake_panel.mat.set_shader_parameter("noise_pattern", snake_panel.noises[index])


func _on_pattern_color_picker_button_color_changed(color: Color) -> void:
	snake_panel.mat.set_shader_parameter("selected_pattern", color)

func _on_fill_color_picker_button_color_changed(color: Color) -> void:
	snake_panel.mat.set_shader_parameter("selected_fill", color)
