/*
 * Copyright (c) 2024 Proton AG
 * This file is part of Proton AG and Proton Pass.
 *
 * Proton Pass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Pass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Pass.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.pass.composecomponents.impl.item.details.sections.identity

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import kotlinx.collections.immutable.ImmutableMap
import kotlinx.collections.immutable.toPersistentList
import kotlinx.datetime.Instant
import proton.android.pass.common.api.Option
import proton.android.pass.commonui.api.Spacing
import proton.android.pass.commonuimodels.api.attachments.AttachmentsState
import proton.android.pass.composecomponents.impl.attachments.AttachmentSection
import proton.android.pass.composecomponents.impl.item.details.PassItemDetailsUiEvent
import proton.android.pass.composecomponents.impl.item.details.PassItemDetailsUiEvent.OnAttachmentEvent
import proton.android.pass.composecomponents.impl.item.details.sections.shared.PassItemDetailsExtraSection
import proton.android.pass.composecomponents.impl.item.details.sections.shared.PassItemDetailsHistorySection
import proton.android.pass.composecomponents.impl.item.details.sections.shared.PassItemDetailsMoreInfoSection
import proton.android.pass.composecomponents.impl.utils.PassItemColors
import proton.android.pass.domain.ItemContents
import proton.android.pass.domain.ItemDiffs
import proton.android.pass.domain.ItemId
import proton.android.pass.domain.ShareId
import proton.android.pass.domain.TotpState
import proton.android.pass.domain.VaultId

@Composable
internal fun PassIdentityItemDetailsSections(
    modifier: Modifier = Modifier,
    itemId: ItemId,
    shareId: ShareId,
    vaultId: VaultId,
    contents: ItemContents.Identity,
    itemColors: PassItemColors,
    itemDiffs: ItemDiffs.Identity,
    lastAutofillOption: Option<Instant>,
    revision: Long,
    createdAt: Instant,
    modifiedAt: Instant,
    shouldDisplayItemHistorySection: Boolean,
    shouldDisplayItemHistoryButton: Boolean,
    attachmentsState: AttachmentsState,
    customFieldTotps: ImmutableMap<Pair<Option<Int>, Int>, TotpState>,
    personalDetailTotps: ImmutableMap<Pair<Option<Int>, Int>, TotpState>,
    addressDetailTotps: ImmutableMap<Pair<Option<Int>, Int>, TotpState>,
    workDetailTotps: ImmutableMap<Pair<Option<Int>, Int>, TotpState>,
    contactDetailTotps: ImmutableMap<Pair<Option<Int>, Int>, TotpState>,
    onEvent: (PassItemDetailsUiEvent) -> Unit
) = with(contents) {
    Column(
        modifier = modifier,
        verticalArrangement = Arrangement.spacedBy(space = Spacing.small)
    ) {
        if (personalDetailsContent.hasPersonalDetails) {
            PassIdentityItemDetailsPersonalSection(
                personalDetailsContent = personalDetailsContent,
                personalDetailTotps = personalDetailTotps,
                itemColors = itemColors,
                itemDiffs = itemDiffs,
                onEvent = onEvent
            )
        }

        if (addressDetailsContent.hasAddressDetails) {
            PassIdentityItemDetailsAddressSection(
                addressDetailsContent = addressDetailsContent,
                addressDetailTotps = addressDetailTotps,
                itemColors = itemColors,
                itemDiffs = itemDiffs,
                onEvent = onEvent
            )
        }

        if (contactDetailsContent.hasContactDetails) {
            PassIdentityItemDetailsContactSection(
                contactDetailsContent = contactDetailsContent,
                contactDetailTotps = contactDetailTotps,
                itemColors = itemColors,
                itemDiffs = itemDiffs,
                onEvent = onEvent
            )
        }

        if (workDetailsContent.hasWorkDetails) {
            PassIdentityItemDetailsWorkSection(
                workDetailsContent = workDetailsContent,
                workDetailTotps = workDetailTotps,
                itemColors = itemColors,
                itemDiffs = itemDiffs,
                onEvent = onEvent
            )
        }

        if (extraSectionContentList.isNotEmpty()) {
            PassItemDetailsExtraSection(
                extraSectionContents = extraSectionContentList.toPersistentList(),
                customFieldTotps = customFieldTotps,
                itemColors = itemColors,
                itemDiffs = itemDiffs,
                onEvent = onEvent
            )
        }

        AttachmentSection(
            attachmentsState = attachmentsState,
            isDetail = true,
            itemColors = itemColors,
            itemDiffs = itemDiffs.attachments,
            onEvent = { onEvent(OnAttachmentEvent(it)) }
        )

        if (shouldDisplayItemHistorySection) {
            PassItemDetailsHistorySection(
                modifier = Modifier.padding(vertical = Spacing.extraSmall),
                lastAutofillAtOption = lastAutofillOption,
                revision = revision,
                createdAt = createdAt,
                modifiedAt = modifiedAt,
                itemColors = itemColors,
                onViewItemHistoryClicked = { onEvent(PassItemDetailsUiEvent.OnViewItemHistoryClick) },
                shouldDisplayItemHistoryButton = shouldDisplayItemHistoryButton
            )
        }

        PassItemDetailsMoreInfoSection(
            itemId = itemId,
            shareId = shareId,
            vaultId = vaultId
        )
    }
}
