/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.features.home.master.ui

import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import proton.android.authenticator.features.home.master.R
import proton.android.authenticator.features.home.master.presentation.HomeMasterState
import proton.android.authenticator.shared.ui.domain.components.buttons.VerticalActionsButtons
import proton.android.authenticator.shared.ui.domain.components.refresh.PullToRefresh
import proton.android.authenticator.shared.ui.domain.theme.Theme
import proton.android.authenticator.shared.ui.domain.theme.ThemePadding
import proton.android.authenticator.shared.ui.domain.theme.ThemeSpacing
import proton.android.authenticator.shared.ui.R as uiR

@Composable
internal fun HomeEmpty(
    state: HomeMasterState.Empty,
    onNewEntryClick: () -> Unit,
    onImportEntriesClick: () -> Unit,
    onEntriesRefreshPull: (Boolean) -> Unit,
    modifier: Modifier = Modifier
) = with(state) {
    val scrollState = rememberScrollState()

    PullToRefresh(
        modifier = modifier,
        isRefreshing = isRefreshing,
        onRefresh = { onEntriesRefreshPull(isSyncEnabled) }
    ) {
        Column(
            modifier = Modifier
                .fillMaxSize()
                .verticalScroll(state = scrollState),
            verticalArrangement = Arrangement.Center
        ) {
            Column(
                modifier = Modifier.offset(y = -ThemeSpacing.Large),
                horizontalAlignment = Alignment.CenterHorizontally,
                verticalArrangement = Arrangement.spacedBy(space = ThemeSpacing.Large)
            ) {
                Image(
                    painter = painterResource(id = uiR.drawable.ic_placeholder_saturn),
                    contentDescription = null
                )

                Column(
                    horizontalAlignment = Alignment.CenterHorizontally,
                    verticalArrangement = Arrangement.spacedBy(space = ThemeSpacing.Small)
                ) {
                    Text(
                        text = stringResource(id = R.string.home_empty_title),
                        textAlign = TextAlign.Center,
                        color = Theme.colorScheme.textNorm,
                        style = Theme.typography.headline
                    )

                    Text(
                        modifier = Modifier.padding(horizontal = ThemePadding.ExtraLarge),
                        text = stringResource(id = R.string.home_empty_description),
                        textAlign = TextAlign.Center,
                        color = Theme.colorScheme.textWeak,
                        style = Theme.typography.bodyRegular
                    )
                }

                VerticalActionsButtons(
                    modifier = Modifier.padding(horizontal = ThemePadding.ExtraLarge),
                    primaryActionText = stringResource(id = R.string.home_empty_action_primary),
                    onPrimaryActionClick = onNewEntryClick,
                    secondaryActionText = stringResource(id = R.string.home_empty_action_import_codes),
                    onSecondaryActionClick = onImportEntriesClick
                )
            }
        }
    }
}
