/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.common

sealed class BuildFlavor(val env: BuildEnv) {
    class Dev(env: BuildEnv) : BuildFlavor(env)
    class Alpha(env: BuildEnv) : BuildFlavor(env)
    class Play(env: BuildEnv) : BuildFlavor(env)
    class Fdroid(env: BuildEnv) : BuildFlavor(env)

    companion object {
        fun from(string: String): BuildFlavor = when (string) {
            "devBlack" -> Dev(BuildEnv.BLACK)
            "devProd" -> Dev(BuildEnv.PROD)
            "alphaBlack" -> Alpha(BuildEnv.BLACK)
            "alphaProd" -> Alpha(BuildEnv.PROD)
            "playBlack" -> Play(BuildEnv.BLACK)
            "playProd" -> Play(BuildEnv.PROD)
            "fdroidBlack" -> Fdroid(BuildEnv.BLACK)
            "fdroidProd" -> Fdroid(BuildEnv.PROD)
            else -> throw UnsupportedOperationException("Unsupported flavour")
        }

        fun BuildFlavor.toValue(): String = when (this) {
            is Dev -> when (this.env) {
                BuildEnv.BLACK -> "devBlack"
                BuildEnv.PROD -> "devProd"
            }
            is Alpha -> when (this.env) {
                BuildEnv.BLACK -> "alphaBlack"
                BuildEnv.PROD -> "alphaProd"
            }
            is Play -> when (this.env) {
                BuildEnv.BLACK -> "playBlack"
                BuildEnv.PROD -> "playProd"
            }
            is Fdroid -> when (this.env) {
                BuildEnv.BLACK -> "fdroidBlack"
                BuildEnv.PROD -> "fdroidProd"
            }
        }
    }
}
