/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.app.ui

import android.app.ComponentCaller
import android.content.Intent
import android.os.Build
import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.viewModels
import androidx.core.view.WindowCompat
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.launch
import proton.android.authenticator.app.handler.RequestReviewHandler
import proton.android.authenticator.app.presentation.MainState
import proton.android.authenticator.app.presentation.MainViewModel
import proton.android.authenticator.business.applock.domain.AppLockState
import proton.android.authenticator.features.shared.usecases.applock.UpdateAppLockStateUseCase
import proton.android.authenticator.navigation.domain.navigators.NavigationNavigator
import proton.android.authenticator.shared.ui.domain.theme.isDarkTheme
import javax.inject.Inject

@AndroidEntryPoint
internal class MainActivity : FragmentActivity() {

    private val viewModel: MainViewModel by viewModels()

    @Inject
    internal lateinit var navigationNavigator: NavigationNavigator

    @Inject
    internal lateinit var updateAppLockStateUseCase: UpdateAppLockStateUseCase

    @Inject
    internal lateinit var requestReviewHandler: RequestReviewHandler

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        viewModel.onRegisterOrchestrators(context = this)

        enableEdgeToEdge()

        lifecycleScope.launch {
            viewModel.requestReview.filterNotNull().collectLatest {
                requestReviewHandler.request(this@MainActivity)
            }
        }

        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest { state ->
                    when (state) {
                        MainState.Loading -> Unit
                        is MainState.Ready -> {
                            setContent {
                                setSecureMode(isSecure = state.isBiometricLockEnabled)

                                isDarkTheme(state.themeType)
                                    .also(::setStatusBarTheme)
                                    .also { isDarkTheme ->
                                        navigationNavigator.NavGraphs(
                                            isDarkTheme = isDarkTheme,
                                            onAskForReview = {
                                                viewModel.askForReviewIfApplicable(state)
                                            },
                                            onFinishLaunching = {
                                                viewModel.setInstallationTimeIfFirstRun(state)
                                            },
                                            onLaunchNavigationFlow = viewModel::onLaunchNavigationFlow
                                        )
                                    }
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onActivityResult(
        requestCode: Int,
        resultCode: Int,
        data: Intent?,
        caller: ComponentCaller
    ) {
        lifecycleScope.launch {
            updateAppLockStateUseCase(state = AppLockState.AuthNotRequired)

            super.onActivityResult(requestCode, resultCode, data, caller)
        }
    }

    private fun setSecureMode(isSecure: Boolean) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            setRecentsScreenshotEnabled(!isSecure)
        }
    }

    private fun setStatusBarTheme(isDarkTheme: Boolean) {
        WindowCompat.getInsetsController(window, window.decorView)
            .also { controller -> controller.isAppearanceLightStatusBars = !isDarkTheme }
    }

}
