// Copyright (c) 2022, Hoylen Sue. All rights reserved. Use of this source code
// is governed by a BSD-style license that can be found in the LICENSE file.
//
// Tests cryptographically secure random number generator enforcement.

library crypt.test;

import 'dart:async';

import 'package:crypt/crypt.dart';
import 'package:test/test.dart';

//################################################################

Future main() async {
  group('cryptographically secure salts', () {
    const key = 'foobar';

    //----------------
    // When a cryptographically secure random number generator MUST be used

    test('mandatory', () {
      try {
        Crypt.cryptographicallySecureSalts = true; // prohibit insecure rnd

        final x = Crypt.sha256(key);
        final y = Crypt.sha256(key);
        expect(x.salt, isNot(equals(y.salt)), reason: 'salt is not random');
        // Don't really have a test to prove it is cryptographically secure
      } on UnsupportedError {
        // This will happen on platforms which don't have a cryptographically
        // secure random number generator. But it probably will never be
        // encountered on the platforms used for testing.
      }
    });

    //----------------
    // When a non-cryptographically secure random number generator can be used

    test('optional', () {
      try {
        Crypt.cryptographicallySecureSalts = false; // allow insecure rnd

        final x = Crypt.sha256(key);
        final y = Crypt.sha256(key);
        expect(x.salt, isNot(equals(y.salt)), reason: 'salt is not random');
      } on UnsupportedError {
        fail('unexpected failure when non-cryptographically random allowed');
      }
    });
  });
}
