/*
 *  Copyright (c) 2022~2025 chr_56
 */

package player.phonograph.ui.modules.main.pages

import player.phonograph.R
import player.phonograph.model.sort.SortMode
import player.phonograph.model.sort.SortRef
import player.phonograph.model.ui.ItemLayoutStyle
import player.phonograph.settings.Keys
import player.phonograph.settings.Setting
import player.phonograph.util.ui.isLandscape
import android.content.Context
import android.content.res.Resources

// todo valid input
sealed class PageDisplayConfig(context: Context) {

    protected val res: Resources = context.resources
    protected val isLandscape: Boolean get() = isLandscape(res)

    abstract var sortMode: SortMode
    abstract val availableSortRefs: Array<SortRef>

    abstract var layout: ItemLayoutStyle
    abstract val availableLayouts: Array<ItemLayoutStyle>

    abstract var gridSize: Int
    abstract val maxGridSize: Int

    abstract var colorFooter: Boolean

    open val allowColoredFooter: Boolean = true
    open val allowRevertSort: Boolean = true

    protected val setting = Setting(context)
}

sealed class ImagePageDisplayConfig(context: Context) : PageDisplayConfig(context) {
    override val availableLayouts: Array<ItemLayoutStyle>
        get() = arrayOf(
            ItemLayoutStyle.LIST,
            ItemLayoutStyle.LIST_EXTENDED,
            ItemLayoutStyle.LIST_NO_IMAGE,
            ItemLayoutStyle.LIST_3L,
            ItemLayoutStyle.LIST_3L_EXTENDED,
            ItemLayoutStyle.LIST_3L_NO_IMAGE,
            ItemLayoutStyle.GRID,
        )
    override val maxGridSize: Int
        get() = if (isLandscape) res.getInteger(R.integer.max_columns_land)
        else res.getInteger(R.integer.max_columns)
}

class SongPageDisplayConfig(context: Context) : ImagePageDisplayConfig(context) {

    override val availableSortRefs: Array<SortRef>
        get() = arrayOf(
            SortRef.SONG_NAME,
            SortRef.ALBUM_NAME,
            SortRef.ARTIST_NAME,
            SortRef.ALBUM_ARTIST_NAME,
            SortRef.COMPOSER,
            SortRef.YEAR,
            SortRef.ADDED_DATE,
            SortRef.MODIFIED_DATE,
            SortRef.DURATION,
        )

    override var layout: ItemLayoutStyle
        get() = if (isLandscape) setting[Keys.songItemLayoutLand].data else setting[Keys.songItemLayout].data
        set(value) {
            if (isLandscape) setting[Keys.songItemLayoutLand].data = value
            else setting[Keys.songItemLayout].data = value
        }
    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.songGridSizeLand].data else setting[Keys.songGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.songGridSizeLand].data = value
            else setting[Keys.songGridSize].data = value
        }
    override var sortMode: SortMode
        get() = setting[Keys.songSortMode].data
        set(value) {
            setting[Keys.songSortMode].data = value
        }
    override var colorFooter: Boolean
        get() = setting[Keys.songColoredFooters].data
        set(value) {
            setting[Keys.songColoredFooters].data = value
        }

}

class AlbumPageDisplayConfig(context: Context) : ImagePageDisplayConfig(context) {

    override val availableSortRefs: Array<SortRef>
        get() = arrayOf(
            SortRef.ALBUM_NAME,
            SortRef.ARTIST_NAME,
            SortRef.YEAR,
            SortRef.SONG_COUNT,
        )

    override var layout: ItemLayoutStyle
        get() = if (isLandscape) setting[Keys.albumItemLayoutLand].data else setting[Keys.albumItemLayout].data
        set(value) {
            if (isLandscape) setting[Keys.albumItemLayoutLand].data = value
            else setting[Keys.albumItemLayout].data = value
        }
    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.albumGridSizeLand].data else setting[Keys.albumGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.albumGridSizeLand].data = value
            else setting[Keys.albumGridSize].data = value
        }
    override var sortMode: SortMode
        get() = setting[Keys.albumSortMode].data
        set(value) {
            setting[Keys.albumSortMode].data = value
        }
    override var colorFooter: Boolean
        get() = setting[Keys.albumColoredFooters].data
        set(value) {
            setting[Keys.albumColoredFooters].data = value
        }

}

class ArtistPageDisplayConfig(context: Context) : ImagePageDisplayConfig(context) {

    override val availableSortRefs: Array<SortRef>
        get() = arrayOf(
            SortRef.ARTIST_NAME,
            SortRef.ALBUM_COUNT,
            SortRef.SONG_COUNT,
        )

    override var layout: ItemLayoutStyle
        get() = if (isLandscape) setting[Keys.artistItemLayoutLand].data else setting[Keys.artistItemLayout].data
        set(value) {
            if (isLandscape) setting[Keys.artistItemLayoutLand].data = value
            else setting[Keys.artistItemLayout].data = value
        }
    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.artistGridSizeLand].data else setting[Keys.artistGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.artistGridSizeLand].data = value
            else setting[Keys.artistGridSize].data = value
        }
    override var sortMode: SortMode
        get() = setting[Keys.artistSortMode].data
        set(value) {
            setting[Keys.artistSortMode].data = value
        }
    override var colorFooter: Boolean
        get() = setting[Keys.artistColoredFooters].data
        set(value) {
            setting[Keys.artistColoredFooters].data = value
        }

}

class PlaylistPageDisplayConfig(context: Context) : PageDisplayConfig(context) {

    override val availableSortRefs: Array<SortRef>
        get() = arrayOf(
            SortRef.DISPLAY_NAME,
            SortRef.PATH,
            SortRef.ADDED_DATE,
            SortRef.MODIFIED_DATE,
        )

    override var layout: ItemLayoutStyle = ItemLayoutStyle.LIST_SINGLE_ROW
    override val availableLayouts: Array<ItemLayoutStyle> get() = arrayOf(ItemLayoutStyle.LIST_SINGLE_ROW)

    override val maxGridSize: Int get() = if (isLandscape) 4 else 2

    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.playlistGridSizeLand].data else setting[Keys.playlistGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.playlistGridSizeLand].data = value
            else setting[Keys.playlistGridSize].data = value
        }
    override var sortMode: SortMode
        get() = setting[Keys.playlistSortMode].data
        set(value) {
            setting[Keys.playlistSortMode].data = value
        }
    override var colorFooter: Boolean = false
    override val allowColoredFooter: Boolean
        get() = false
}

class GenrePageDisplayConfig(context: Context) : PageDisplayConfig(context) {

    override val availableSortRefs: Array<SortRef>
        get() = arrayOf(
            SortRef.DISPLAY_NAME,
            SortRef.SONG_COUNT,
        )

    override var layout: ItemLayoutStyle = ItemLayoutStyle.LIST_NO_IMAGE
    override val availableLayouts: Array<ItemLayoutStyle> get() = arrayOf(ItemLayoutStyle.LIST_NO_IMAGE)

    override val maxGridSize: Int get() = if (isLandscape) 6 else 4

    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.genreGridSizeLand].data else setting[Keys.genreGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.genreGridSizeLand].data = value
            else setting[Keys.genreGridSize].data = value
        }
    override var sortMode: SortMode
        get() = setting[Keys.genreSortMode].data
        set(value) {
            setting[Keys.genreSortMode].data = value
        }
    override var colorFooter: Boolean = false
    override val allowColoredFooter: Boolean
        get() = false
}

class FolderPageDisplayConfig(context: Context) : PageDisplayConfig(context) {
    override val availableSortRefs: Array<SortRef> get() = arrayOf(SortRef.DISPLAY_NAME) //todo: support SortRef.ADDED_DATE, SortRef.MODIFIED_DATE, SortRef.SIZE)

    override val availableLayouts: Array<ItemLayoutStyle>
        get() = arrayOf(
            ItemLayoutStyle.LIST,
            ItemLayoutStyle.LIST_EXTENDED,
            ItemLayoutStyle.LIST_3L,
            ItemLayoutStyle.LIST_3L_EXTENDED,
        )
    override var layout: ItemLayoutStyle
        get() =
            if (isLandscape) {
                setting[Keys.folderItemLayoutLand].data
            } else {
                setting[Keys.folderItemLayout].data
            }
        set(value) {
            if (isLandscape) {
                setting[Keys.folderItemLayoutLand].data = value
            } else {
                setting[Keys.folderItemLayout].data = value
            }
        }

    override val maxGridSize: Int get() = if (isLandscape) 4 else 2
    override var gridSize: Int
        get() = if (isLandscape) setting[Keys.folderGridSizeLand].data else setting[Keys.folderGridSize].data
        set(value) {
            if (value <= 0) return
            if (isLandscape) setting[Keys.folderGridSizeLand].data = value
            else setting[Keys.folderGridSize].data = value
        }

    override var sortMode: SortMode
        get() = setting[Keys.collectionSortMode].data
        set(value) {
            setting[Keys.collectionSortMode].data = value
        }
    override var colorFooter: Boolean = false
    override val allowColoredFooter: Boolean get() = false

}