/*
 *  Copyright (c) 2022~2023 chr_56
 */

package player.phonograph.ui.modules.web

import player.phonograph.R
import player.phonograph.ui.compose.components.HorizontalTextItem
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ColumnScope
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentWidth
import androidx.compose.material.DropdownMenu
import androidx.compose.material.DropdownMenuItem
import androidx.compose.material.Icon
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.material.TextButton
import androidx.compose.material.TextField
import androidx.compose.material.TextFieldDefaults
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.outlined.ArrowDropDown
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp


@Composable
fun BaseSearchBox(
    modifier: Modifier,
    title: String,
    target: @Composable () -> Unit,
    onSearch: () -> Unit,
    fields: @Composable ColumnScope.() -> Unit,
) {
    Column(modifier.padding(vertical = 8.dp)) {
        Text(
            text = title,
            modifier = Modifier.padding(horizontal = 12.dp, vertical = 8.dp),
            fontWeight = FontWeight.Black,
            style = MaterialTheme.typography.h5
        )
        HorizontalTextItem(label = stringResource(R.string.label_target)) {
            target()
        }
        fields()
        TextButton(
            onClick = {
                onSearch()
            },
            modifier = Modifier
                .padding(horizontal = 12.dp)
                .align(Alignment.End)
        ) {
            Text(stringResource(R.string.action_search))
        }
    }
}

@Composable
@Suppress("UNUSED_PARAMETER")
fun SearchTextBox(current: String, modifier: Modifier = Modifier, hint: String? = null, onUpdate: (String) -> Unit) {
    TextField(
        value = current,
        onValueChange = onUpdate,
        modifier = Modifier.fillMaxWidth(),
        colors = TextFieldDefaults.textFieldColors(
            backgroundColor = MaterialTheme.colors.background,
            textColor = MaterialTheme.colors.onBackground,
        ),
        trailingIcon = {
            Icon(
                Icons.Default.Clear,
                contentDescription = stringResource(id = R.string.action_reset),
                modifier = Modifier
                    .padding(8.dp)
                    .clickable {
                        onUpdate.invoke("")
                    }
            )
        },
        placeholder = {
            if (!hint.isNullOrEmpty()) {
                val textStyle = TextStyle(
                    fontSize = 14.sp,
                    color = MaterialTheme.colors.onBackground.copy(alpha = 0.3f)
                )
                Text(hint, style = textStyle)
            }
        },
    )
}

@Composable
fun <T> Target(
    all: Collection<T>,
    text: @Composable (T) -> String,
    current: T,
    modifier: Modifier = Modifier,
    onUpdate: (T) -> Unit,
) {
    var selected by remember { mutableStateOf(current) }
    var expanded by remember { mutableStateOf(false) } // initial value

    Row(
        modifier.clickable {
            expanded = !expanded
        },
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.Top,
    ) {
        Text(
            text = text(selected),
            modifier = Modifier
                .weight(1f)
                .padding(horizontal = 16.dp, vertical = 8.dp)
        )
        Icon(Icons.Outlined.ArrowDropDown, null, modifier = Modifier.padding(8.dp))
        DropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false },
            modifier = Modifier.wrapContentWidth()
        ) {
            for (item in all) {
                DropdownMenuItem(
                    onClick = {
                        selected = item
                        expanded = false
                        onUpdate(item)
                    }
                ) {
                    Text(
                        text = text(item),
                        modifier = Modifier.wrapContentWidth()
                    )
                }
            }
        }
    }
}