/*
 * Copyright (C) 2016 Player One
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package player.efis.pfd;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.location.GpsStatus.Listener;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.opengl.GLSurfaceView;
import android.os.Build;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.WindowManager;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.List;

import player.efis.common.ADSBContainer;
import player.efis.common.AircraftData;
import player.efis.common.AirspaceClass;
import player.efis.common.DemGTOPO30;
import player.efis.common.SensorComplementaryFilter;
import player.efis.common.WxRadarMap;
import player.efis.common.module_t;
import player.efis.common.orientation_t;
import player.efis.common.prefs_t;
import player.ulib.UMath;
import player.ulib.UNavigation;
import player.ulib.Unit;

public class PFDMainActivity extends EFISMainActivity implements Listener, SensorEventListener, LocationListener
{
    public static final String PREFS_NAME = R.string.app_name + ".prefs";
    //private MainSurfaceView mGLView;
    private EFISSurfaceView mGLView;

    /**
     * method checks to see if app is currently set as default launcher
     * @return boolean true means currently set as default, otherwise false
     */
    private boolean isMyAppLauncherDefault()
    {
        final IntentFilter filter = new IntentFilter(Intent.ACTION_MAIN);
        filter.addCategory(Intent.CATEGORY_HOME);

        List<IntentFilter> filters = new ArrayList<IntentFilter>();
        filters.add(filter);

        final String myPackageName = getPackageName();
        List<ComponentName> activities = new ArrayList<ComponentName>();
        final PackageManager packageManager = getPackageManager();

        packageManager.getPreferredActivities(filters, activities, null);

        for (ComponentName activity : activities) {
            if (myPackageName.equals(activity.getPackageName())) {
                return true;
            }
        }
        return false;
    }

    /**
     * method starts an intent that will bring up a prompt for the user
     * to select their default launcher. It comes up each time it is
     * detected that our app is not the default launcher
     */
    private void launchAppChooser()
    {
        //PackageManager packageManager = this.getPackageManager();
        //ComponentName componentName = new ComponentName(this, player.efis.pfd.FakeLauncherActivity.class);
        //packageManager.setComponentEnabledSetting(componentName, PackageManager.COMPONENT_ENABLED_STATE_ENABLED, PackageManager.DONT_KILL_APP);

        Intent intent = new Intent(Intent.ACTION_MAIN);
        intent.addCategory(Intent.CATEGORY_HOME);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        //startActivity(intent);
        startActivity(Intent.createChooser(intent, "Choose Default Launcher"));
    }


    //
    //  Add the action bar buttons
    //
    //Menu menu;
    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        // Inflate the menu; this adds items to the action bar if it is present.
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main_menu, menu);
        return super.onCreateOptionsMenu(menu);
    }

    /*
    @Override
    public boolean onPrepareOptionsMenu (Menu menu)
     {
         //MenuItem item = menu.findItem(R.id.menu_my_item);
         MenuItem item = menu.findItem(1);
         menu.getItem(1).setEnabled(false);
         if (true) {
             item.setEnabled(true);
             item.getIcon().setAlpha(255);
         } else {
             // disabled
             item.setEnabled(false);
             item.getIcon().setAlpha(130);
         }
         return true;
    }*/

    @Override
    public void onBackPressed()
    {
        if (mGLView.mRenderer.fatFingerActive == true) {
            mGLView.mRenderer.fatFingerActive = false;
            mGLView.mRenderer.setSpinnerParams();
        }
        else openOptionsMenu();
    }

    @Override
    public void onOptionsMenuClosed(Menu menu) {
        super.onOptionsMenuClosed(menu);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            // Workaround for https://issuetracker.google.com/issues/315761686
            invalidateOptionsMenu();
        }
    }


    // This method is called once the menu is selected
    @Override
    public boolean onOptionsItemSelected(MenuItem item)
    {
        switch (item.getItemId()) {
            case R.id.settings:
                // Launch settings activity
                Intent i = new Intent(this, PrefSettings.class);
                startActivity(i);
                break;
            case R.id.manage:
                // Launch manage activity
                Intent j = new Intent(this, PrefManage.class);
                startActivity(j);
                break;
            case R.id.quit:
                // Quit the app
                if (bLockedMode == false) {
                    if (isMyAppLauncherDefault()) launchAppChooser();
                    finish();
                    //System.exit(0); // This is brutal, it does not exit gracefully
                }
                else Toast.makeText(this, "Locked Mode: Active", Toast.LENGTH_SHORT).show();
                break;
            // more code...
            default:
                return super.onOptionsItemSelected(item);
        }
        return true;
    }

    // This code will catch the actual keypress.
    // for now we will leave the menu bar in case it is needed later
    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event)
    {
        return super.onKeyDown(keyCode, event);
    }

    // Force homepage
    /*
    @Override
    protected void onPause() {
        super.onPause();
        //ActivityManager activityManager = (ActivityManager) getApplicationContext().getSystemService(Context.ACTIVITY_SERVICE);
        //activityManager.moveTaskToFront(getTaskId(), 0);
    }
    */


    @Override
    public void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        // Keep the screen on
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);

        // Create a GLSurfaceView instance and set it
        // as the ContentView for this Activity
        createView();

        /*
        // Always cold start the app with the default (PFD) page

        SharedPreferences settings = PreferenceManager.getDefaultSharedPreferences(getBaseContext());
        SharedPreferences.Editor editor = settings.edit();

        if (includeMultiInSwipe) activeModule = module_t.toInteger(module_t.CFD);  // Let the app always start at the default (PFD)
        else activeModule = module_t.toInteger(module_t.PFD);  // Let the app always start at the default (PFD)
        editor.putString("activeModule", String.valueOf(activeModule));
        editor.commit();
        */

        // Get the version number of the app
        PackageInfo pInfo = null;
        try {
            pInfo = getPackageManager().getPackageInfo(getPackageName(), 0);
        }
        catch (NameNotFoundException e) {
            e.printStackTrace();
        }
        //String version = pInfo.versionName;
        //Toast.makeText(this, "Kwik EFIS version: " + version, Toast.LENGTH_SHORT).show();

        try {
            mSensorManager = (SensorManager) getSystemService(Activity.SENSOR_SERVICE);
            registerSensorManagerListeners();
        }
        catch (Exception e) {
            Toast.makeText(this, "Hardware compatibility issue", Toast.LENGTH_LONG).show();
        }

        // testing for lightweight -- may or may not use
        sensorComplementaryFilter = new SensorComplementaryFilter();

        // Preferences
        PreferenceManager.setDefaultValues(this, R.xml.settings, false);
        PreferenceManager.setDefaultValues(this, R.xml.manage, false);

        // Set the window to be full brightness
        WindowManager.LayoutParams layout = getWindow().getAttributes();
        layout.screenBrightness = -1f;  // 1f = full bright 0 = selected
        getWindow().setAttributes(layout);

        // Initial lazy loads
        lazyLoadDemGTOPO30DemBuffer(gps_lat, gps_lon);  // Do this before the databases; Sets DemGTOPO30.lat0 and DemGTOPO30.lon0,
        lazyLoadGpxDatabase(gps_lat, gps_lon);
        lazyLoadOpenAirspaceDatabase(gps_lat, gps_lon);

        // This should never happen but we catch and force it to something valid it just in case
        if (mGLView.mRenderer.mSelWptName.length() != 4) mGLView.mRenderer.mSelWptName = "ZZZZ";
        if (mGLView.mRenderer.mAltSelName.length() != 5) mGLView.mRenderer.mSelWptName = "00000";

        createMediaPlayer();

        // Overall the device is now ready.
        // The individual elements will be enabled or disabled by the location provided
        // based on availability
        mGLView.setServiceableDevice();
    }

    void createView()
    {
        EFISRenderer Renderer;
        Renderer = new PFDRenderer(this); //0

        mGLView = new EFISSurfaceView(this);
        mGLView.mRenderer = Renderer;
        mGLView.setRenderer((GLSurfaceView.Renderer) Renderer);    // from constructor
        mGLView.setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY); // from constructor
        mGLView.setTheme(colorTheme);
        setContentView(mGLView);

        // Apply the saved settings to this view
        restorePersistentVars();
    }

    protected void pauseActivity()
    {
        mGLView.clearBannerMsg();
        mGLView.onPause();
    }

    protected void resumeActivity()
    {
        mGLView.clearBannerMsg();
        mGLView.onResume();
    }


    //
    // Sensor methods
    //
    @Override
    public void onAccuracyChanged(Sensor sensor, int accuracy)
    {
        switch (sensor.getType()) {
            case Sensor.TYPE_GRAVITY:
                break;

            case Sensor.TYPE_MAGNETIC_FIELD:
                // SENSOR_STATUS_ACCURACY_HIGH
                // SENSOR_STATUS_ACCURACY_MEDIUM
                // SENSOR_STATUS_ACCURACY_LOW
                // SENSOR_STATUS_UNRELIABLE
                break;
        }
    }

    @Override
    public void onSensorChanged(SensorEvent event)
    {
        switch (event.sensor.getType()) {
            case Sensor.TYPE_ACCELEROMETER:
                sensorComplementaryFilter.setAccel(event.values);
                accelPitch = -sensorComplementaryFilter.getPitch();
                accelRoll = -sensorComplementaryFilter.getRoll();
                break;

            case Sensor.TYPE_GYROSCOPE:
                sensorComplementaryFilter.setGyro(event.values);
                break;

            case Sensor.TYPE_MAGNETIC_FIELD:
                //b2b2 sensorFusion.setMagnet(event.values);
                break;

            case Sensor.TYPE_ORIENTATION:
                orientationAzimuth = event.values[0];
                orientationPitch = -event.values[2];
                orientationRoll = -event.values[1];
                break;

            case Sensor.TYPE_PRESSURE:
                // altitude = mSensorManager.getAltitude(SensorManager.PRESSURE_STANDARD_ATMOSPHERE, event.values[0]);
                break;
        }
    }


    @Override
    public void onLocationChanged(Location location)
    {
        if (bSimulatorActive) return;
        //if (bUseGdl90Ahrs) return;   // it is possible Android has better GPS than the ADSB device

        gps_lat = (float) location.getLatitude();
        gps_lon = (float) location.getLongitude();
        gps_agl = DemGTOPO30.calculateAgl(gps_lat, gps_lon, gps_altitude);

        if (location.hasSpeed()) {
            gps_speed = location.getSpeed();
            if (gps_speed == 0) gps_speed = 0.01f;  // nip div zero issues in the bud
            mGLView.setServiceableAsi();
            hasSpeed = true;
        }
        else {
            mGLView.setUnServiceableAsi();
            hasSpeed = false;
        }

        if (location.hasAltitude()) {
            gps_altitude = (float) location.getAltitude();
            gps_rateOfClimb = calculateRateOfClimb(gps_altitude);
            mGLView.setServiceableAlt();
        }
        else {
            mGLView.setUnServiceableAlt();
        }

        if (location.hasBearing()) {
            if (gps_speed > 2) {
                gps_course = (float) Math.toRadians(location.getBearing());
                gps_rateOfTurn = calculateRateOfTurn(gps_course);
                mGLView.setServiceableDi();
                mGLView.setServiceableRose();
            }
            else {
                gps_rateOfTurn = 0;
            }
        }
        else {
            gps_course = 0;
            gps_rateOfTurn = 0;
            mGLView.setUnServiceableDi();
            mGLView.setUnServiceableRose();
        }

        if (location.hasSpeed() && location.hasBearing()) {
            hasGps = true;
            mGLView.setServiceableAh();
        }
        else {
            hasGps = false;
            mGLView.setUnServiceableAh();
        }
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras)
    {
        if (!LocationManager.GPS_PROVIDER.equals(provider)) {
            return;
        }
        // do something
    }

    protected void savePersistentVars()
    {
        // We need an Editor object to make preference changes.
        // All objects are from android.context.Context
        // Save persistent preferences
        SharedPreferences settings = getSharedPreferences(PREFS_NAME, 0);
        SharedPreferences.Editor editor = settings.edit();
        editor.putString("WptSelName", mGLView.mRenderer.mSelWptName);
        editor.putString("WptSelComment", mGLView.mRenderer.mSelWptComment);
        editor.putFloat("WptSelLat", mGLView.mRenderer.mSelWptLat);
        editor.putFloat("WptSelLon", mGLView.mRenderer.mSelWptLon);
        editor.putFloat("mAltSelValue", mGLView.mRenderer.mAltSelValue);
        editor.putString("mAltSelName", mGLView.mRenderer.mAltSelName);
        editor.putFloat("mObsValue", mGLView.mRenderer.mObsValue);
        //editor.putFloat("mMapZoom", mGLView.mRenderer.mMapZoom);

        // Save last known location
        editor.putFloat("GpsLat", gps_lat);
        editor.putFloat("GpsLon", gps_lon);

        editor.putBoolean("includeMultiInSwipe", includeMultiInSwipe);

        // Commit the edits
        editor.commit();
    }

    private void restorePersistentVars()
    {
        // We need an Editor object to make preference changes.
        // All objects are from android.context.Context
        // Restore persistent preferences
        SharedPreferences settings = getSharedPreferences(PREFS_NAME, 0);
        mGLView.mRenderer.mSelWptName = settings.getString("WptSelName", "ZZZZ");
        mGLView.mRenderer.mSelWptComment = settings.getString("WptSelComment", "Null Island");
        mGLView.mRenderer.mSelWptLat = settings.getFloat("WptSelLat", 00f);
        mGLView.mRenderer.mSelWptLon = settings.getFloat("WptSelLon", 00f);
        mGLView.mRenderer.mAltSelValue = settings.getFloat("mAltSelValue", 0f);
        mGLView.mRenderer.mAltSelName = settings.getString("mAltSelName", "00000");
        mGLView.mRenderer.mObsValue = settings.getFloat("mObsValue", 0f);
        //mGLView.mRenderer.mMapZoom = settings.getFloat("mMapZoom", 100f);  //5f // Zoom multiplier for map. 1 (max out) to 200 (max in)

        // Restore last known location
        gps_lat = settings.getFloat("GpsLat", 0);
        gps_lon = settings.getFloat("GpsLon", 0);

        includeMultiInSwipe = settings.getBoolean("includeMultiInSwipe", false);
        if (includeMultiInSwipe) activeModule = module_t.toInteger(module_t.CFD);
    }

    /*
    // This must be implemented otherwise the older
    // systems does not get seem to get updates.
    @Override
    public void onGpsStatusChanged(int state)
    {
        setGpsStatus();
    }

    @Override
    public void onProviderEnabled(String provider)
    {
        Toast.makeText(this, "Enabled new provider " + provider, Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onProviderDisabled(String provider)
    {
        Toast.makeText(this, "Disabled provider " + provider, Toast.LENGTH_SHORT).show();
    }
	*/
    // end location abs ------------------------


    protected void setUserPrefs()
    {
        SharedPreferences settings = PreferenceManager.getDefaultSharedPreferences(getBaseContext());

        mGLView.setPrefs(prefs_t.DEM, settings.getBoolean("displayDEM", false));
        mGLView.setPrefs(prefs_t.WX, settings.getBoolean("displayWX", false));
        mGLView.setPrefs(prefs_t.TAPE, settings.getBoolean("displayTape", true));
        mGLView.setPrefs(prefs_t.MIRROR, settings.getBoolean("displayMirror", false));
        mGLView.setPrefs(prefs_t.INFO_PAGE, settings.getBoolean("infoPage", true));
        mGLView.setPrefs(prefs_t.REMOTE_INDICATOR, settings.getBoolean("displayRMI", true));
        mGLView.setPrefs(prefs_t.FLIGHT_DIRECTOR, settings.getBoolean("displayFlightDirector", false));
        mGLView.setPrefs(prefs_t.CCIP, settings.getBoolean("displayCCIP", false));

        // Only used in PFD
        mGLView.setPrefs(prefs_t.AH_COLOR, settings.getBoolean("displayAHColor", true));
        mGLView.setPrefs(prefs_t.HITS, settings.getBoolean("displayHITS", false));

        // Only used in MFD
        mGLView.setPrefs(prefs_t.AIRSPACE, settings.getBoolean("displayAirspace", true));
        AirspaceClass.A = settings.getBoolean("classA", true);
        AirspaceClass.B = settings.getBoolean("classB", true);
        AirspaceClass.C = settings.getBoolean("classC", true);
        AirspaceClass.D = settings.getBoolean("classD", true);
        AirspaceClass.E = settings.getBoolean("classE", true);
        AirspaceClass.F = settings.getBoolean("classF", true);
        AirspaceClass.G = settings.getBoolean("classG", true);
        AirspaceClass.P = settings.getBoolean("classP", true);
        AirspaceClass.R = settings.getBoolean("classR", true);
        AirspaceClass.Q = settings.getBoolean("classQ", true);
        AirspaceClass.CTR = settings.getBoolean("classCTR", true);
        mGLView.setPrefs(prefs_t.RANGE_RINGS, settings.getBoolean("rangeRings", false));

        bLockedMode = settings.getBoolean("lockedMode", false);
        bSwipeActive = settings.getBoolean("swipeActive", true);

        sensorBias = Float.valueOf(settings.getString("sensorBias", "0.0f"));  // ""0.15f"

        // If we changed the local/global status force a Gpx reload
        // Note the menu option is to *enable* the local gpx mode. It is off by default.
        boolean rv = settings.getBoolean("dynamicGpxDatabase", false);
        if (bDynamicGpx != rv) {
            bDynamicGpx = rv;
            mGLView.setBannerMsg(true, "LOADING DATABASE");
            lazyLoadGpxDatabase(gps_lat, gps_lon);
        }

        // If we changed to Demo mode, use the current GPS as seed location
        if (bSimulatorActive != settings.getBoolean("simulatorActive", false)) {
            if (!UNavigation.isNullLatLon(gps_lat, gps_lon)) {
                _gps_lon = gps_lon;
                _gps_lat = gps_lat;
            }
        }

        bSimulatorActive = settings.getBoolean("simulatorActive", false);

        // The external ADS-B device, Stratux etc.
        bUseGdl90Traffic = settings.getBoolean("gdl90TrafficActive", false);
        bUseGdl90Ahrs = settings.getBoolean("gdl90AhrsActive", false);  // use the GPS and AHRS from the ADS-B device if available.
        // We may need some more logic here.
        //if (bUseGdl90Ahrs) bGdl90Active = true;

        bHudMode = settings.getBoolean("displayMirror", false);
        bWeatherActive = settings.getBoolean("displayWX", false);

        // If the aircraft is changed, update the paramaters
        String s = settings.getString("AircraftModel", "RV8");
        AircraftData.setAircraftData(s); //mGLView.mRenderer.setAircraftData(s);  // refactored  to static model

        // landscape / portrait mode toggle
        bLandscapeMode = settings.getBoolean("landscapeMode", false);
        if (bLandscapeMode) {
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);
            mGLView.mRenderer.Layout = EFISRenderer.layout_t.LANDSCAPE;
        }
        else {
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
            mGLView.mRenderer.Layout = EFISRenderer.layout_t.PORTRAIT;
        }
        bLandscapeMode = settings.getBoolean("landscapeMode", false);

        // If we changed display schemes, a color gamma rec-calc is required
        int _colorTheme = Integer.valueOf(settings.getString("colorTheme", "0"));
        if (colorTheme != _colorTheme) {
            colorTheme = _colorTheme; // handle the recursive call
            createView();
            mGLView.setTheme(colorTheme);
            mGLView.invalidate();
            mGLView.setServiceableDevice();
            setUserPrefs();  // make a recursive call
        }

        activeModule = Integer.valueOf(settings.getString("activeModule", "0"));
        mGLView.setActiveModule(module_t.fromInteger(activeModule));  // set the new active module

        // If the user selected CFD, include it in future swipes, otherwise exclude it by default
        SharedPreferences.Editor editor = settings.edit();
        if (activeModule == module_t.toInteger(module_t.PFD))  includeMultiInSwipe = false;
        if (activeModule == module_t.toInteger(module_t.MFD))  includeMultiInSwipe = false;
        if (activeModule == module_t.toInteger(module_t.CFD))  includeMultiInSwipe = true;

        editor.putBoolean("includeMultiInSwipe", includeMultiInSwipe);
        editor.commit();  // Commit the changes
    }

    //
    // Flight Path Vector (the birdie)
    //
    protected void updateFPV()
    {
        float fpvX = 0, fpvY = 0;

        fpvX = (float) filterfpvX.runningAverage(Math.toDegrees(Math.atan2(-gyro_rateOfTurn * 100.0f, gps_speed))); // a point 100m ahead of nose
        fpvY = (float) filterfpvY.runningAverage(Math.toDegrees(Math.atan2(gps_rateOfClimb, gps_speed)));    // simple RA of the two velocities

        mGLView.setFPV(fpvX, fpvY); // need to clean this up
    }

    //
    // Force a blank screen and no birdie
    //
    protected void forceBlankScreen()
    {
        rollValue = 0;
        pitchValue = -270;
        mGLView.setFPV(180, 180);
    }

    /*private boolean isNetworkAvailable()
    {
        ConnectivityManager connectivityManager = (ConnectivityManager) getSystemService(this.CONNECTIVITY_SERVICE);
        //NetworkInfo activeNetworkInfo = connectivityManager != null ? connectivityManager.getActiveNetworkInfo() : null;
        //return activeNetworkInfo != null && activeNetworkInfo.isConnected();

        if (connectivityManager != null) return connectivityManager.getActiveNetworkInfo().isConnected();
        else return false;
    }*/

    //
    // Stratux AHRS handler
    //
    // Note: this function is a bit messy ... Also refer to the parent in EFISMainActivity
    //
    protected int handleGdl90Ahrs()
    {
        int rv = super.handleGdl90Ahrs();

        switch (rv) {
            case GDL90_OK:
                hasGps = true;
                hasSpeed = true;

                mGLView.setServiceableDevice();
                mGLView.setServiceableDi();
                mGLView.setServiceableAsi();
                if (gps_altitude <= -305) mGLView.setUnServiceableAlt(); // Note: -305m (-1000ft) indicates ERROR
                else mGLView.setServiceableAlt();
                mGLView.setServiceableAh();
                mGLView.setServiceableMap();
                mGLView.setDisplayAirport(true);
                updateFPV();
                mGLView.clearBannerMsg();
                mGLView.setStatusMsg(" ");
                break;

            case GDL90_TASK:
                mGLView.setStatusMsg("GDL90 TASK FAIL");
                break;

            case GDL90_SERVICE:
                mGLView.setStatusMsg("GDL90 SERVICE FAIL");
                break;

            case GDL90_DEVICE:
                // no pulse
                mGLView.setStatusMsg("GDL90 DEVICE FAIL");
                break;

            case GDL90_BATTERY:
                // low battery
                mGLView.setStatusMsg("GDL90 BATTERY FAIL");
                break;

            case GDL90_WIFI:
                // No Wifi
                mGLView.setStatusMsg("GDL90 WIFI FAIL");
                break;

            case GDL90_AHRS:
                // if the ADS-B device does not support AHRS or it has failed
                mGLView.setStatusMsg("GDL90 AHRS FAIL");
                break;

            case GDL90_GPS:
                // if the ADS-B device does not support AHRS or it has failed
                mGLView.setStatusMsg("GDL90 GPS FAIL");
                break;
        }
        return rv;
    }

    //
    // Android  handler
    //
    protected boolean handleAndroidAhrs()
    {
        float[] gyro = new float[3]; // gyroscope vector, rad/sec
        float[] accel = new float[3]; // accelerometer vector, m/s^2

        //
        // Read the Sensors
        //
        if (bLandscapeMode) {
            if (bHudMode)
                sensorComplementaryFilter.setOrientation(orientation_t.HORIZONTAL_LANDSCAPE);
            else sensorComplementaryFilter.setOrientation(orientation_t.VERTICAL_LANDSCAPE);
        }
        else {
            if (bHudMode)
                sensorComplementaryFilter.setOrientation(orientation_t.HORIZONTAL_PORTRAIT);
            else sensorComplementaryFilter.setOrientation(orientation_t.VERTICAL_PORTRAIT);
        }

        sensorComplementaryFilter.getGyro(gyro);      // Use the gyroscopes for the attitude
        sensorComplementaryFilter.getAccel(accel);    // Use the accelerometer for G and slip

        if (bLandscapeMode) {
            gyro_rateOfTurn = (float) filterRateOfTurnGyro.runningAverage(-gyro[0]);
            slipValue = (float) filterSlip.runningAverage(Math.toDegrees(Math.atan2(accel[1], accel[0]))); //?
        }
        else {
            gyro_rateOfTurn = (float) filterRateOfTurnGyro.runningAverage(-gyro[1]);
            slipValue = (float) -filterSlip.runningAverage(Math.toDegrees(Math.atan2(accel[0], accel[1])));
        }

        loadfactor = sensorComplementaryFilter.getLoadFactor();
        loadfactor = filterG.runningAverage(loadfactor);

        //
        // Calculate the augmented bank angle and also the flight path vector
        //
        float deltaA, fpvX = 0, fpvY = 0;
        if (hasGps) {
            //mGLView.setStatusMsg(" ");
            mGLView.setServiceableDevice();
            mGLView.setServiceableMap();
            if (gps_speed > 5) {
                // Testing shows that reasonable value is sensorBias of 75% gps and 25% gyro on most older devices,
                // if the gyro and accelerometer are good quality and stable, use sensorBias of 1.0 (100%)
                // sensorBias of 0 uses GPS exclusively
                rollValue = sensorComplementaryFilter.calculateBankAngle((sensorBias) * gyro_rateOfTurn + (1 - sensorBias) * gps_rateOfTurn, gps_speed);
                pitchValue = sensorComplementaryFilter.calculatePitchAngle(gps_rateOfClimb, gps_speed);

                // the Flight Path Vector (FPV)
                updateFPV();
                mGLView.setDisplayAirport(true);
            }
            else {  // m/s
                // taxi mode
                rollValue = 0;
                pitchValue = 0;
            }

            // Add a "Couch mode"
            if (gps_speed < 1) {  // m/s
                // stationary/couch mode - Could be useful to drive simulator?
                pitchValue = accelPitch;
                rollValue = accelRoll;
            }
        }
        else {
            // No GPS no speed ... no idea what the AH is
            mGLView.setStatusMsg("GPS FAIL");
            // forceBlankScreen();  // may be better to just let the RedX deal with it
            mGLView.setUnServiceableAh();
            // We don't handle anything else here since it is all already done in onLocationChanged
        }

        // for debug - set to true
        if (false) {
            hasGps = true;          //debug
            hasSpeed = true;        //debug
            gps_speed = 3;//60;     //m/s debug
            gps_rateOfClimb = 1.0f; //m/s debug
        }
        // end debug
        return true;
    }


    protected void updateDEM()
    {
        mGLView.clearBannerMsg(); // clear any banners

        //
        // Handle the DEM buffer.
        // Load new data to the buffer when the horizon gets close to the edge or
        // if we have gone off the current tile.
        //
        float dem_dme = UNavigation.calcDme(DemGTOPO30.lat0, DemGTOPO30.lon0, gps_lat, gps_lon);

        //
        // Load new data into the buffer when the horizon gets close to the edge
        // and see if we are stuck on null island or even on the tile
        //

        // Departure gets very small near the pole - Causes a display problem
        // If we are close the arctic circle (arbitrarily) dispense with the DEM_HORIZON
        float dem_veil = dem_dme + (Math.abs(gps_lat) > 60 ? 0 : DemGTOPO30.DEM_HORIZON);
        float dem_threshold = UNavigation.calcDeparture(DemGTOPO30.BUFX / 4, gps_lat);

        //Log.d("kwik", "gps_lat= " + gps_lat + " gps_lon= " + gps_lon +  " dem_dme= " + dem_dme + " dem_veil= " + dem_veil + " dem_threshold= " + dem_threshold );

        if ( (dem_veil > dem_threshold)
            || ((dem_dme != 0) && (DemGTOPO30.isOnTile(gps_lat, gps_lon) == false)) ) {

            if (bDynamicGpx == true) {
                mGLView.setBannerMsg(true, "LOADING DATABASE");
                lazyLoadGpxDatabase(gps_lat, gps_lon);
            }

            mGLView.setBannerMsg(true, "LOADING TERRAIN");
            //int rv = DemGTOPO30.loadDemBuffer(gps_lat, gps_lon);
            lazyLoadDemGTOPO30DemBuffer(gps_lat, gps_lon);

            mGLView.setBannerMsg(true, "LOADING AIRSPACE");
            //OpenAirspace.loadDatabase(gps_lat, gps_lon);
            lazyLoadOpenAirspaceDatabase(gps_lat, gps_lon);

            mGLView.clearBannerMsg();  // clear any banners

            // Handle errors
            if (DemGTOPO30.demLastError != DemGTOPO30.DEM_OK) {
                switch (DemGTOPO30.demLastError) {
                    case DemGTOPO30.DEM_SYN_NOT_INSTALLED:
                        mGLView.setBannerMsg(true, "DATAPAC " + DemGTOPO30.getRegionDatabaseName(gps_lat, gps_lon) + " MISSING");
                        break;
                    case DemGTOPO30.DEM_TERRAIN_ERROR:
                        mGLView.setBannerMsg(true, "DATAPAC " + DemGTOPO30.getRegionDatabaseName(gps_lat, gps_lon) + " ERROR");
                        break;
                }
            }
            updateWX();  // This is a useful place to catch large movements and sync the WX
        }
    }

    protected void updateWX()
    {
        time.setToNow();

        // Only update if active and older than 10 minutes 1000*60*10 ms
        //if (bWeatherActive && (time.toMillis(true) - wxTimeStamp) > 600000) {
        {
            mGLView.clearBannerMsg(); // clear any banners
            mGLView.setBannerMsg(true, "LOADING WEATHER");

            // Load from OpenWeatherMap
            updateWeather();
            wxTimeStamp = time.toMillis(true);
            mGLView.clearBannerMsg(); // clear any banners
        }
    }

    protected void updateTraffic()
    {
        super.updateTraffic();
    }

    //-------------------------------------------------------------------------
    // Effectively the main execution loop. updateEFIS will get called when
    // something changes, eg a sensor has new data or new gps fix becomes available.
    //

    //private static Time time2 = new Time();    // Time class
    private static long time_b, _time_b;   // used for silence timeout

    //
    // Refresh rate is 25 fps, see EFISMainActivity.onCreate.FPS
    //
    protected void updateEFIS()
    {
        ctr++;
        if (ctr % 100 == 0 ) {
            // Every 4 sec (1000/25*100)
            mGLView.setStatusMsg(" ");
        }

        //
        // Mode handlers
        //

        mGLView.clearSimulatorActive();  // Clear the simulator splash - Do not use mGLView.clearBannerMsg();

        //
        // AHRS
        //
        if (bSimulatorActive) {
            // Simulator handler
            mGLView.setSimulatorActive(true, "SIMULATOR");
            Simulate();
            // Set the GPS flag to true and
            // make all the instruments serviceable
            mGLView.setServiceableDevice();
            mGLView.setServiceableDi();
            mGLView.setServiceableAsi();
            mGLView.setServiceableAlt();
            mGLView.setServiceableAh();
            mGLView.setServiceableMap();
            mGLView.setDisplayAirport(true);
        }
        else if (bUseGdl90Ahrs) {
            // Stratux/GDL90 handler
            int rv = handleGdl90Ahrs();

            // We need a fully functioning GDL90 device if we want to use GPS or AHRS
            if ((rv == GDL90_OK)) {
                mGLView.setServiceableDevice();
                mGLView.setServiceableDi();
                mGLView.setServiceableAsi();
                mGLView.setServiceableAlt();
                mGLView.setServiceableAh();
            }
            else {
                mGLView.setUnServiceableDevice();
                mGLView.setUnServiceableDi();
                mGLView.setUnServiceableAsi();
                mGLView.setUnServiceableAlt();
                mGLView.setUnServiceableAh();
            }
        }
        else {
            // Internal android handler
            handleAndroidAhrs();

            // Apply a little filtering to the pitch, bank (only for Android, not Stratux)
            pitchValue = filterPitch.runningAverage(pitchValue);
            rollValue = filterRoll.runningAverage(UNavigation.compassRose180(rollValue));
        }

        //
        // Get the battery percentage
        //
        float batteryPct = getRemainingBattery();

        // for debug - set to true
        if (false) {
            hasGps = true;          //debug
            hasSpeed = true;        //debug
            gps_speed = 3;//60;     //m/s debug
            gps_rateOfClimb = 1.0f; //m/s debug
            gps_course = (float) Math.toRadians(1); // debug
        }
        // end debug

        float gps_dme = mGLView.mRenderer.getSelWptDme(); // in nm

        //
        // Pass the values to mGLView for updating
        //
        String s; // general purpose string

        mGLView.setPitch(pitchValue);                             // in degrees
        mGLView.setRoll(rollValue);                               // in degrees
        mGLView.setGForce(loadfactor);                            // in gunits
        mGLView.setSlip(slipValue);                               // in degrees
        mGLView.setVSI((int) Unit.MeterPerSecond.toFeetPerMinute(gps_rateOfClimb));  // in fpm
        //mGLView.setTurn((float) Math.toDegrees((sensorBias) * gyro_rateOfTurn + (1 - sensorBias) * gps_rateOfTurn)); // in deg/sec
        mGLView.setTurn((float) Math.toDegrees(gps_rateOfTurn)); // in deg/sec  ... Just use GPS
        mGLView.setHeading((float) Math.toDegrees(gps_course));   // in degrees
        mGLView.setALT((int) Unit.Meter.toFeet(gps_altitude));    // in Feet
        mGLView.setAGL((int) Unit.Meter.toFeet(gps_agl));         // in Feet
        mGLView.setASI(Unit.MeterPerSecond.toKnots(gps_speed));   // in knots
        mGLView.setLatLon(gps_lat, gps_lon);

        mGLView.setBatteryPct(batteryPct);                        // in percentage

        //
        // Handle traffic
        //
        if (bUseGdl90Traffic && (mGdl90Task != null)) {
            if (mGdl90Task.isDeviceRunning()) {
                // Note: the EFISRenderer.renderAncillaryDetails relies on the text "GDL-90" to determine color.
                if (bUseGdl90Ahrs) mGLView.setActiveDevice("GDL90+ (" + getCurrentSsid() + ")");
                else mGLView.setActiveDevice("GDL90 (" + getCurrentSsid() + ")");

                ADSBContainer.clearTrafficList();
                mGLView.setTargets(mGdl90Task.getTargetList()); // ADSB traffic list
            }
            else {
                mGLView.setActiveDevice("GDL90 (" + getCurrentSsid() + ")");
                mGLView.setStatusMsg("GDL90 ADSB FAIL");
            }
        }
        else {
            mGLView.setActiveDevice("INTERNET");

            //if ( (!ADSBContainer.dataLinkOK) || (!hasCellularInternet()) ) {
            //if ( (!ADSBContainer.dataLinkOK) || (!isNetworkAvailable()) ) {
            //if (!hasCellularInternet())  {

            if (!ADSBContainer.dataLinkOK) {
                mGLView.setStatusMsg("NETWORK FAIL");
            }
            else {
                ADSBContainer.setLatLon(gps_lat, gps_lon);
                mGLView.setTargets(ADSBContainer.getTargetList()); // OpenSky traffic list
            }
        }

        // Handle weather
        WxRadarMap.setLatLon(gps_lat, gps_lon);

        //
        // Audio cautions and messages
        //
        // If sound is muted. reset after 30 seconds
        time.setToNow();
        time_b = time.toMillis(true);
        long deltaT = time_b - _time_b;
        if (deltaT > mGLView.mRenderer.MUTE_TIMEOUT) {
            mGLView.mRenderer.mute = false;
            _time_b = time_b;
        }

         if (hasGps && !mGLView.isMuted()) {
            try {
                // We have new traffic
                // Implement a suitable detection and reporting strategy
                if (mGLView.getProximityAlert()) {
                    if (!mpCautionTraffic.isPlaying()) mpCautionTraffic.start();
                    mGLView.setProximityAlert(false);
                }

                // We are stalling, advise captain "Crash" of his imminent flight emergency
                if (hasSpeed
                        && (gps_speed < 3 + AircraftData.Vs0 / 2) // m/s, warn 3 m/s before stall
                        && (gps_speed > 3)                        // m/s, warn only when faster than 3 m/s
                        && (gps_agl > 10)) {                      // meters
                    if (!mpStall.isPlaying()) mpStall.start();
                }

                // Sigh ... Now, we are plummeting to the ground, inform the prick on the stick
                if (gps_rateOfClimb < -10) { // m ~ 2000 fpm //gps_rateOfClimb * 196.8504f for fpm
                    if (!mpSinkRate.isPlaying()) mpSinkRate.start();
                }

                // We are at risk of becoming a wet spot somewhere on terra firma
                if (DemGTOPO30.demDataValid) {
                    // Play the "caution terrain" song above Vx,
                    // below Vx and close to the destination we assume "Ace" is attempting to land
                    if ((gps_speed > AircraftData.Vx)  // m/s
                            && (gps_agl > 0)
                            && (gps_agl < 100)  // in meter
                            && (gps_dme > 2)) { // nm
                        if (!mpCautionTerrian.isPlaying()) mpCautionTerrian.start();
                    } // caution terrain

                    // Play the "five hundred" song when descending through 500ft
                    // and closer than 2nm from dest
                    if ((_gps_agl > 152.4f)  // 500ft
                            && (gps_agl <= 152.4f) // 500ft
                            && (gps_dme < 2)) {
                        if (!mpFiveHundred.isPlaying()) mpFiveHundred.start();
                    }
				} // DemGTOPO30 required options
            }
            catch (IllegalStateException e) {
                e.printStackTrace();
            }
        }
        _gps_agl = gps_agl; // save the previous height agl
    }


    protected void Simulate()
    {
        SimulateCfd();
        super.Simulate();
    }

    protected void SimulateCfd()
    {
        pitchValue = -sensorComplementaryFilter.getPitch();
        rollValue = -sensorComplementaryFilter.getRoll();

        pitchValue = /*0.05f * (float) Math.random()*/ + 0.75f * UMath.clamp(mGLView.mRenderer.commandPitch, -3, 3);
        rollValue = /*0.05f * (float) Math.random()*/ + 0.75f * mGLView.mRenderer.commandRoll;
    }

    protected void SimulateMfd()
    {
        int target_agl = 3000; //ft

        pitchValue = -sensorComplementaryFilter.getPitch();
        rollValue = -sensorComplementaryFilter.getRoll();

        pitchValue = /*0.125f * (float) Math.random()*/ - 0.75f * UMath.clamp(gps_agl - Unit.Feet.toMeter(target_agl), -3, 3);
        rollValue = /*1.125f * (float) Math.random()*/ + 0.75f * mGLView.mRenderer.commandRoll;
    }

}


