/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package player.efis.pfd;

import android.content.Context;
import android.content.SharedPreferences;
import android.opengl.GLSurfaceView;
import android.preference.PreferenceManager;
import android.text.format.Time;
import android.view.MotionEvent;
import android.widget.Toast;

import java.util.LinkedList;

import player.efis.common.Gdl90WiFiTask;
import player.efis.common.module_t;
import player.efis.common.prefs_t;

//
// A view container where OpenGL ES graphics can be drawn on screen.
// This view can also be used to capture touch events, such as a user
// interacting with drawn objects.
//
public class EFISSurfaceView extends GLSurfaceView
{
   public EFISRenderer mRenderer;  // normally this would be private but we want to access the sel wpt from main activity

    protected static Time time = new Time();    // Time class
    private static long time_a, _time_a;

    public EFISSurfaceView(Context context)
    {
        super(context);

        ///*
        // Create an OpenGL ES 2.0 context.
        setEGLContextClientVersion(2);

        // Set the Renderer for drawing on the GLSurfaceView

        // NOTE : we choose the renderer  later ...
        //-- mRenderer = new CFDRenderer(context); // = new MyGLRenderer();  --b2
        //-- setRenderer(mRenderer);

        // Render the view only when there is a change in the drawing data
        // -- setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        //*/

        /*
        Then just use this Renderer in your GLSurfaceView

        glSurfaceView.setEGLContextClientVersion(1);
        glSurfaceView.setRenderer(new BitmapRenderer(getResources()));
        glSurfaceView.setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        */
        /*
        mRenderer = new CFDRenderer(context); // = new MyGLRenderer();  --b2
        setEGLContextClientVersion(1);  //1
        setRenderer(new BitmapRenderer(getResources()));
        setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        //*/

        /*
        // Create an OpenGL ES 2.0 context.
        setEGLContextClientVersion(2);  //2

        // Set the Renderer for drawing on the GLSurfaceView
        mRenderer = new CFDRenderer(context);
        setRenderer(mRenderer);

        // Render the view only when there is a change in the drawing data
        setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        //*/
    }

    //private final float TOUCH_SCALE_FACTOR = 180.0f / 320;
    private float mPreviousX;
    private float mPreviousY;
    private final int MIN_DISTANCE = 250;
    public boolean swipe = false;

    @Override
    public boolean onTouchEvent(MotionEvent e)
    {
        // MotionEvent reports input details from the touch screen
        // and other input controls. In this case, you are only
        // interested in events where the touch position changed.

        float x = e.getX();
        float y = e.getY();

        switch (e.getAction()) {
            /* We don't use movement messages, maybe later ...
            case MotionEvent.ACTION_MOVE:

                float dx = x - mPreviousX;
                float dy = y - mPreviousY;

                // reverse direction of rotation above the mid-line
                if (y > getHeight() / 2) {
                    dx = dx * -1;
                }

                // reverse direction of rotation to left of the mid-line
                if (x < getWidth() / 2) {
                    dy = dy * -1;
                }
                requestRender();
                break;
            */

            case MotionEvent.ACTION_DOWN:
                mRenderer.setActionDown(x, y);

                float deltaX = x - mPreviousX;
                float deltaY = y - mPreviousY;
                requestRender();

                mPreviousX = x;
                mPreviousY = y;

                // handle a double tap - < 300 ms
                time.setToNow();
                time_a = time.toMillis(true);
                long deltaT = time_a - _time_a;
                if (deltaT > 0) _time_a = time_a;
                // double tap
                if (!mRenderer.fatFingerActive && (deltaT < 300) && (Math.abs(deltaX + deltaY) < MIN_DISTANCE/10)) {
                    if (!mRenderer.isAutoZoomActive()) {
                        Toast.makeText(getContext(), "Auto Zoom ON", Toast.LENGTH_SHORT).show();
                        setAutoZoomActive(true);
                    }
                }
                break;

            case MotionEvent.ACTION_UP:
                mRenderer.setActionUp(x, y);
                deltaX = x - mPreviousX;
                deltaY = y - mPreviousY;

                // Handle a long press- > 2 seconds
                time.setToNow();
                time_a = time.toMillis(true);
                deltaT = time_a - _time_a;

                if (deltaT > 2000) {
                    // Long press - 3 seconds
                    Toast.makeText(getContext(), "Level AHRS", Toast.LENGTH_SHORT).show();
                    Gdl90WiFiTask.doCageAhrs();
                }
                else if (Math.abs(deltaY) > MIN_DISTANCE) {
                    // Handle swipes
                    // vertical swipes
                    setAutoZoomActive(false);
                    if (deltaY < 0) {
                        // swipe up
                        zoomIn();
                    }
                    else {
                        // swipe down
                        zoomOut();
                    }
                }
                else if (Math.abs(deltaX) > MIN_DISTANCE) {
                    // horizontal swipes
                    swipe = true;
                    SharedPreferences settings = PreferenceManager.getDefaultSharedPreferences(mRenderer.context);
                    int activeModule = Integer.valueOf(settings.getString("activeModule", "0"));
                    boolean bIncludeMultiInSwipe = settings.getBoolean("includeMultiInSwipe", false);
                    int maxPage, minPage;

                    if (bIncludeMultiInSwipe) {
                        minPage = 2;
                        maxPage = 3;  // include CFD
                    }
                    else {
                        minPage = 0;
                        maxPage = 2;  // exclude CFD
                    }

                    boolean bSwipeActive = settings.getBoolean("swipeActive", true);
                    SharedPreferences.Editor editor = settings.edit();

                    if (bSwipeActive) {
                        if (deltaX < 0) {
                            // swipe right
                            activeModule++;
                            if (activeModule > maxPage) {
                                activeModule = minPage;
                            }
                        }
                        else {
                            activeModule--;
                            if (activeModule < minPage) {
                                activeModule = maxPage;
                            }
                        }

                        // Commit the changes
                        editor.putString("activeModule", String.valueOf(activeModule));
                        editor.commit();

                        //mRenderer.mActiveViewModule = module_t.fromInteger(activeModule);
                        setActiveModule(module_t.fromInteger(activeModule));
                    }
                }
                else if (!isMuted()) {
                    // single tap and release
                    setMute(true);    // mute sound on screen tap, must be reset by the sound handler if needed .
                    Toast.makeText(getContext(), "Mute "+ mRenderer.MUTE_TIMEOUT/1000 + " for seconds", Toast.LENGTH_SHORT).show();
                }
                else {
                    // consider as something else - a screen tap for example
                }
                break;
        }
        requestRender();
        return true;
    }

    // Pitch
    public void setPitch(float degrees)
    {
        mRenderer.setPitch(degrees);
        requestRender();
    }

    // Roll / Bank
    public void setRoll(float degrees)
    {
        mRenderer.setRoll(degrees);
        requestRender();
    }

    // Heading / Course indicator
    public void setHeading(float degrees)
    {
        mRenderer.setHeading(degrees);
        requestRender();
    }

    // Altimeter
    public void setALT(int value)
    {
        mRenderer.setAlt(value);
        requestRender();
    }

    // Radio Altimeter (agl)
    public void setAGL(int value)
    {
        mRenderer.setAGL(value);
        requestRender();
    }


    // Air Speed Indicator
    public void setASI(float value)
    {
        mRenderer.setASI(value);
        requestRender();
    }

    // Vertical Speed Indicator
    public void setVSI(int value)
    {
        mRenderer.setVSI(value);
        requestRender();
    }

    // FLight Path Vector
    public void setFPV(float fpvX, float fpvY)
    {
        mRenderer.setFPV(fpvX, fpvY);
        requestRender();
    }

    // G force
    public void setGForce(float gunits)
    {
        mRenderer.setGForce(gunits);
        requestRender();
    }

    // Percentage battery remaining
    public void setBatteryPct(float value)
    {
        mRenderer.setBatteryPct(value);
        requestRender();
    }

    // The amount of Slip
    public void setSlip(float slip)
    {
        mRenderer.setSlip(slip);
        requestRender();
    }

    // Automatic Waypoint
    public void setWPT(String wpt)
    {
        mRenderer.setAutoWptName(wpt);
        requestRender();
    }

    // The DME - Distance Measuring Equipment
    public void setDME(float dme)
    {
        mRenderer.setAutoWptDme(dme);
        requestRender();
    }

    // Ground Speed
    public void setGS(float gs)
    {
        //mRenderer.setGS(gs);
        //requestRender();
    }


    // Turn Rate
    public void setTurn(float rot)
    {
        mRenderer.setTurn(rot);
        requestRender();
    }

    //
    // Red X's
    //

    // The entire EFIS
    public void setUnServiceableDevice()
    {
        mRenderer.setUnServiceableDevice();
        requestRender();
    }

    public void setServiceableDevice()
    {
        mRenderer.setServiceableDevice();
        requestRender();
    }

    // Artificial Horizon
    public void setUnServiceableAh()
    {
        mRenderer.setUnServiceableAh();
        requestRender();
    }

    public void setServiceableAh()
    {
        mRenderer.setServiceableAh();
        requestRender();
    }

    // Artificial Horizon
    public void setUnServiceableRose()
    {
        mRenderer.setUnServiceableRose();
        requestRender();
    }

    public void setServiceableRose()
    {
        mRenderer.setServiceableRose();
        requestRender();
    }

    // Moving Map
    public void setUnServiceableMap()
    {
        mRenderer.setUnServiceableMap();
        requestRender();
    }

    public void setServiceableMap()
    {
        mRenderer.setServiceableMap();
        requestRender();
    }

    // Altimeter
    public void setUnServiceableAlt()
    {
        mRenderer.setUnServiceableAlt();
        requestRender();
    }

    public void setServiceableAlt()
    {
        mRenderer.setServiceableAlt();
        requestRender();
    }

    // Airspeed
    public void setUnServiceableAsi()
    {
        mRenderer.setUnServiceableAsi();
        requestRender();
    }

    public void setServiceableAsi()
    {
        mRenderer.setServiceableAsi();
        requestRender();
    }

    // Direction Indicator
    public void setUnServiceableDi()
    {
        mRenderer.setUnServiceableDi();
        requestRender();
    }

    public void setServiceableDi()
    {
        mRenderer.setServiceableDi();
        requestRender();
    }

    public void setLatLon(float lat, float lon)
    {
        mRenderer.setLatLon(lat, lon);
        requestRender();
    }

    /*void setDisplayFPV(boolean display)
    {
        mRenderer.setDisplayFPV(display);
        requestRender();
    }*/

    public void setDisplayAirport(boolean display)
    {
        mRenderer.setDisplayAirport(display);
        requestRender();
    }

    public void setDisplayAirspace(boolean display)
    {
        mRenderer.setDisplayAirspace(display);
        requestRender();
    }

    public void setStatusMsg(String msg)
    {
        mRenderer.setStatusMsg(msg);
        requestRender();
    }

    public void setBannerMsg(boolean cal, String msg)
    {
        mRenderer.setBannerMsg(cal, msg);
        requestRender();
    }

    public void clearBannerMsg()
    {
        mRenderer.setBannerMsg(false, " ");
        requestRender();
    }

    public void setSimulatorActive(boolean status, String msg)
    {
        mRenderer.setSimulatorActive(status, msg);
        requestRender();
    }

    public void clearSimulatorActive()
    {
        mRenderer.setSimulatorActive(false, " ");
        requestRender();
    }

    public void setPrefs(prefs_t pref, boolean value)
    {
        mRenderer.setPrefs(pref, value);
        requestRender();
    }

    public void setTheme(int value)
    {
        if (value == 2) mRenderer.setThemeGreen();
        else if (value == 1)  mRenderer.setThemeLight();
        else mRenderer.setThemeDark(); // the default
        requestRender();
    }

    //
    // ADSB targets
    //
    /*public void setTargets(StratuxWiFiTask Stratux)
    {
        mRenderer.setTargets(Stratux);
        requestRender();
    }*/

    //public void setTargets(OpenSkyTask OpenSky)
    public void setTargets(LinkedList<String> targetlist)
    {
        mRenderer.setTargets(targetlist);
        requestRender();
    }

    //
    // Active Device
    //
    public void setActiveDevice(String device)
    {
        mRenderer.setActiveDevice(device);
        requestRender();
    }

    public void setActiveModule(module_t module)
    {
        mRenderer.setActiveModule(module);
        requestRender();
    }

    public void setMapZoom(float zoom)
    {
        mRenderer.setMapZoom(zoom);
        requestRender();
    }

    public void zoomIn()
    {
        mRenderer.zoomIn();
        requestRender();
    }

    public void zoomOut()
    {
        mRenderer.zoomOut();
        requestRender();
    }

    public void setAutoZoomActive(boolean active)
    {
        mRenderer.setAutoZoomActive(active);
        requestRender();
    }


    // Traffic proximity alert
    public boolean getProximityAlert()
    {
        return mRenderer.proximityAlert;
    }

    public void setProximityAlert(boolean alert)
    {
        mRenderer.proximityAlert = alert;
    }

    public void setMute(boolean mute)
    {
        mRenderer.mute = mute;
    }

    public boolean isMuted()
    {
        return mRenderer.mute;
    }
}
