/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package player.gles20;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import android.opengl.GLES20;

/**
 * A two-dimensional line for use as a drawn object in OpenGL ES 2.0.
 */
public class Line {


	// This matrix member variable provides a hook to manipulate
	// the coordinates of the objects that use this vertex shader
	private final String vertexShaderCode =
	"uniform mat4 uMVPMatrix;" +
	"attribute vec4 vPosition;" +
	"void main() {" +
	"  gl_Position = uMVPMatrix * vPosition;" +
	"}"; // the matrix must be included as a modifier of gl_Position

	private final String fragmentShaderCode =
			"precision mediump float;" +
			"uniform vec4 vColor;" +
			"void main() {" +
			"  gl_FragColor = vColor;" +
			"}";

	private final FloatBuffer mVertexBuffer;
	private final int mProgram;
	private int mPositionHandle;
	private int mColorHandle;
	private int mMVPMatrixHandle;

	// number of coordinates per vertex in this array
	static final int COORDS_PER_VERTEX = 3;
	//static float LineCoords[] = { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f };
	static float LineCoords[] = { -1.0f, 0.50f, 0.0f, 0.5f, 0.0f, 0.0f };
	static float LineWidth = 1.0f;

	private final int VertexCount = LineCoords.length / COORDS_PER_VERTEX;
	private final int VertexStride = COORDS_PER_VERTEX * 4; // 4 bytes per
															// vertex

	// Set color with red, green, blue and alpha (opacity) values
	float color[] = { 0.0f, 0.0f, 0.0f, 1.0f };

    /**
     * Utility method for compiling a OpenGL shader.
     * <p>
     * <p><strong>Note:</strong> When developing shaders, use the checkGlError()
     * method to debug shader coding errors.</p>
     *
     * @param type       - Vertex or fragment shader type.
     * @param shaderCode - String containing the shader code.
     * @return - Returns an id for the shader.
     */
    public static int loadShader(int type, String shaderCode)
    {
        // create a vertex shader type (GLES20.GL_VERTEX_SHADER)
        // or a fragment shader type (GLES20.GL_FRAGMENT_SHADER)
        int shader = GLES20.glCreateShader(type);

        // add the source code to the shader and compile it
        GLES20.glShaderSource(shader, shaderCode);
        GLES20.glCompileShader(shader);

        return shader;
    }


	public Line() {
		// initialize vertex byte buffer for shape coordinates
		ByteBuffer bb = ByteBuffer.allocateDirect(LineCoords.length * 4); // (number of coordinate values * 4 bytes per float)
		// use the device hardware's native byte order
		bb.order(ByteOrder.nativeOrder());

		// create a floating point buffer from the ByteBuffer
		mVertexBuffer = bb.asFloatBuffer();
		// add the coordinates to the FloatBuffer
		mVertexBuffer.put(LineCoords);
		// set the buffer to read the first coordinate
		mVertexBuffer.position(0);

		// prepare shaders and OpenGL program
		int vertexShader = loadShader(GLES20.GL_VERTEX_SHADER, vertexShaderCode);
		int fragmentShader = loadShader(GLES20.GL_FRAGMENT_SHADER, fragmentShaderCode);

		mProgram = GLES20.glCreateProgram(); // create empty OpenGL ES Program
		GLES20.glAttachShader(mProgram, vertexShader); // add the vertex shader  to program
		GLES20.glAttachShader(mProgram, fragmentShader); // add the fragment  shader to program
		GLES20.glLinkProgram(mProgram); // creates OpenGL ES program  executables
	}

	public void SetVerts(float v0, float v1, float v2, float v3, float v4, float v5) {
		LineCoords[0] = v0;
		LineCoords[1] = v1;
		LineCoords[2] = v2;
		LineCoords[3] = v3;
		LineCoords[4] = v4;
		LineCoords[5] = v5;

		mVertexBuffer.put(LineCoords);
		// set the buffer to read the first coordinate
		mVertexBuffer.position(0);
	}

	public void SetColor(float red, float green, float blue, float alpha) {
		color[0] = red;
		color[1] = green;
		color[2] = blue;
		color[3] = alpha;
	}

	public void SetWidth(float width)
	{
		LineWidth = width;
	}

    /**
     * Encapsulates the OpenGL ES instructions for drawing this shape.
     *
     * @param mvpMatrix - The Model View Project matrix in which to draw
     * this shape.
     */
	public void draw(float[] mvpMatrix)
	{
		// Add program to OpenGL ES environment
		GLES20.glUseProgram(mProgram);

		// get handle to vertex shader's vPosition member
		mPositionHandle = GLES20.glGetAttribLocation(mProgram, "vPosition");

		// Enable a handle to the triangle vertices
		GLES20.glEnableVertexAttribArray(mPositionHandle);

		// Prepare the line coordinate data
		GLES20.glVertexAttribPointer(mPositionHandle, COORDS_PER_VERTEX, GLES20.GL_FLOAT, false, VertexStride, mVertexBuffer);

		// get handle to fragment shader's vColor member
		mColorHandle = GLES20.glGetUniformLocation(mProgram, "vColor");

		// Set color for drawing the line
		GLES20.glUniform4fv(mColorHandle, 1, color, 0);

        // Note:
        // float lineWidthRange[] = {0.0f, 0.0f};
        // GLES20.glGetFloatv(GLES20.GL_ALIASED_LINE_WIDTH_RANGE, FloatBuffer.wrap(lineWidthRange));
        // Maximum supported line width is in lineWidthRange[1].
        //

		GLES20.glLineWidth(LineWidth);

		// get handle to shape's transformation matrix
		mMVPMatrixHandle = GLES20.glGetUniformLocation(mProgram, "uMVPMatrix");
		//EFISRenderer.checkGlError("glGetUniformLocation");

		// Apply the projection and view transformation
		GLES20.glUniformMatrix4fv(mMVPMatrixHandle, 1, false, mvpMatrix, 0);
		//EFISRenderer.checkGlError("glUniformMatrix4fv");

		// Draw the line
		GLES20.glDrawArrays(GLES20.GL_LINES, 0, VertexCount);

		// Disable vertex array
		GLES20.glDisableVertexAttribArray(mPositionHandle);
	}
}
