use bevy::{ecs::system::RunSystemOnce, prelude::*};
use terratactician_expandoria::{
    game::{
        gamemodes::{campaign::CampaignModeConfig, challenge::ChallengeGameState, GameMode},
        metrics::Metrics,
        recorder::Record,
        replayer::{start_replay, Replay},
        report::Warnings,
    },
    log::{LoggingConf, LoggingController},
    AppBuilder, AppState, DisplayMode, GameConfig,
};

fn setup_replay(register_logger: bool, content: &str) -> bevy::app::App {
    let logctr = if register_logger {
        Some(LoggingController::init(LoggingConf::default()))
    } else {
        None
    };
    let mut app = AppBuilder {
        display_mode: DisplayMode::Headless,
        #[cfg(feature = "graphics")]
        enable_config_io: false,
        logging_controller: logctr,
    }
    .build();
    app.finish();
    app.cleanup();

    let mut config = GameConfig {
        skip_recorder: true,
        skip_seed: true,
        skip_mode: true,
        ..default()
    };

    let replay = serde_json::from_str::<Record>(&content).expect("Error while parsing replay.");

    config.selected_gamemode = replay.mode.clone();
    match replay.mode.clone() {
        GameMode::Challenge => {
            let challenge_config = replay
                .challenge
                .clone()
                .expect("Error while parsing replay: missing challenge seed");

            app.insert_resource(challenge_config);
        }
        GameMode::Creative => (),
        GameMode::Campaign => {
            let wrapper = replay
                .campaign
                .clone()
                .expect("Error while parsing replay: missing campaign config");

            app.insert_resource(CampaignModeConfig { level: wrapper });
        }
        GameMode::Zen => {
            let zen_config = replay
                .zen
                .clone()
                .expect("Error while parsing replay: missing challenge seed");

            app.insert_resource(zen_config);
        }
    };

    app.insert_resource(Replay(replay));
    app.add_systems(
        OnEnter(AppState::AppMenu),
        (|mut commands: Commands| {
            commands.queue(move |w: &mut World| w.run_system_once(start_replay).unwrap());
        })
        .run_if(run_once),
    );

    app.insert_resource(config);
    app
}

const REPLAY1: &str = include_str!("files/replay1.json");

#[test]
fn replay_game() {
    let mut app = setup_replay(true, REPLAY1);

    while ChallengeGameState::GameOver
        != app
            .world()
            .get_resource::<State<ChallengeGameState>>()
            .map_or(ChallengeGameState::BetweenRounds, |s| **s)
    {
        if let Some(state) = app.should_exit() {
            panic!("Game exited premature: {state:?}");
        }
        app.update();
    }

    let world = app.world();
    let metrics = world.resource::<Metrics>();
    let replay = world
        .resource::<Replay>()
        .metrics
        .clone()
        .expect("Error while gathering replay, from game.");
    assert_eq!(*metrics, replay, "Replay was not reproducible.");
    let warnings = world.resource::<Warnings>();
    assert_eq!(
        *warnings,
        Warnings::default(),
        "Invalid actions where played, while running the replayer."
    );
}
