use std::fmt::Display;

use http::Uri;
use wasm_bindgen::prelude::*;

use super::{WsError, WsMessage};

#[wasm_bindgen(module = "/src/websocket/ws.js")]
extern "C" {
    pub fn ws_open(url: &str) -> i32;
    pub fn ws_write(id: u32, msg: &str) -> bool;
    pub fn ws_read(id: u32) -> String;
    pub fn ws_state(id: u32) -> i32;
    pub fn ws_available(id: u32) -> i32;
    pub fn ws_close(id: u32);
}

pub fn from_state_id(id: i32) -> Result<Option<()>, WsError> {
    use WsError::*;
    match id {
        0 => Ok(None),
        1 => Ok(Some(())),
        2 => Err(Closed),
        3 => Err(Url),
        _ => Err(Failed),
    }
}

pub struct WebSocket(u32);

impl WebSocket {
    pub fn read<R>(&mut self) -> Result<Option<R>, WsError>
    where
        R: TryFrom<WsMessage>,
        R::Error: Display,
    {
        let state = from_state_id(ws_state(self.0))?;
        let Some(_) = state else {
            return Err(WsError::AlreadyFailed);
        };

        let queue_len = ws_available(self.0);
        if queue_len < 0 {
            return Err(WsError::AlreadyFailed);
        }
        if queue_len == 0 {
            return Ok(None);
        }
        let msg = ws_read(self.0);
        match R::try_from(msg) {
            Ok(data) => Ok(Some(data)),
            Err(err) => Err(WsError::ParseError(format!("{err}"))),
        }
    }

    pub fn write<S: Into<WsMessage>>(&mut self, msg: S) -> Result<(), WsError> {
        let state = from_state_id(ws_state(self.0))?;
        let Some(_) = state else {
            return Err(WsError::AlreadyFailed);
        };

        if ws_write(self.0, (msg.into()).as_str()) {
            Ok(())
        } else {
            Err(WsError::Failed)
        }
    }

    pub fn close(&mut self) -> Result<(), WsError> {
        ws_close(self.0);
        Ok(())
    }
}

impl Drop for WebSocket {
    fn drop(&mut self) {
        let _ = self.close();
    }
}

pub struct WebSocketConnector(u32);
impl WebSocketConnector {
    pub fn new(url: Uri) -> Result<Self, WsError> {
        match ws_open(&url.to_string()) {
            x if x < 0 => Err(WsError::Url),
            x => Ok(Self(x as u32)),
        }
    }
    pub fn step_connect(&mut self) -> Result<Option<WebSocket>, WsError> {
        from_state_id(ws_state(self.0)).map(|state| {
            state.map(|_| {
                let id = self.0;
                self.0 = u32::MAX;
                WebSocket(id)
            })
        })
    }
}

impl Drop for WebSocketConnector {
    fn drop(&mut self) {
        ws_close(self.0);
    }
}
