use std::io;
use std::time::Duration;

use std::fmt::Display;
use std::net::{SocketAddr, ToSocketAddrs};
use std::vec::IntoIter;

use http::{Request, Uri};
use mio::event::Event;
use mio::net::TcpStream;
use mio::{Events, Interest, Poll, Token};
use tungstenite::client::IntoClientRequest;
use tungstenite::handshake::MidHandshake;
use tungstenite::{ClientHandshake, HandshakeError, Message};

use super::{WsError, WsMessage};

pub struct WebSocket {
    socket: tungstenite::WebSocket<TcpStream>,
    _poll: Poll,
}

impl From<tungstenite::error::Error> for WsError {
    fn from(value: tungstenite::error::Error) -> Self {
        match value {
            tungstenite::Error::AlreadyClosed => WsError::Closed,
            tungstenite::Error::ConnectionClosed => WsError::Closed,
            tungstenite::Error::Url(_) => WsError::Url,
            tungstenite::Error::Io(_) => WsError::Url,
            _ => WsError::Failed,
        }
    }
}

impl From<io::Error> for WsError {
    fn from(_: io::Error) -> Self {
        // maybe improve errors
        WsError::Url
    }
}

impl From<Option<tungstenite::protocol::frame::CloseFrame>> for WsError {
    fn from(_value: Option<tungstenite::protocol::frame::CloseFrame>) -> Self {
        WsError::Closed
    }
}

impl WebSocket {
    pub fn read<R>(&mut self) -> Result<Option<R>, WsError>
    where
        R: TryFrom<WsMessage>,
        R::Error: Display,
    {
        match self.socket.read() {
            Ok(Message::Text(text)) => match R::try_from(text.to_string()) {
                Ok(data) => Ok(Some(data)),
                Err(err) => Err(WsError::ParseError(format!("{err}"))),
            },
            Ok(Message::Close(cf)) => Err(cf.into()),
            Err(tungstenite::Error::Io(io_error))
                if io_error.kind() == io::ErrorKind::WouldBlock =>
            {
                Ok(None)
            }
            Err(e) => {
                // maybe a smart way to handle the other errors
                Err(e.into())
            }
            _ => Ok(None),
        }
    }

    pub fn write<S: Into<WsMessage>>(&mut self, msg: S) -> Result<(), WsError> {
        self.socket.write(Message::Text(msg.into().into()))?;
        self.socket.flush()?;
        Ok(())
    }

    pub fn close(&mut self) -> Result<(), WsError> {
        self.socket.close(None).map_err(WsError::from)
    }
}

//pub struct ConnectionAttempt(Cell<Option<ConnectionStatus>>);

pub struct WebSocketConnector(Option<ConnectionAttemptInternal>);

struct ConnectionAttemptInternal {
    status: ConnectionStatus,
    poll: Poll,
    events: Events,
    addrs: IntoIter<SocketAddr>,
    req: Request<()>,
}

impl WebSocketConnector {
    const CLIENT: Token = Token(0);
    pub fn new(uri: Uri) -> Result<Self, WsError> {
        let req = uri.clone().into_client_request()?;
        let mut addrs = (
            uri.host().unwrap_or("127.0.0.1"),
            uri.port_u16().unwrap_or(7738),
        )
            .to_socket_addrs()?;
        let Some(addr) = addrs.next() else {
            return Err(WsError::Url);
        };
        let mut stream = TcpStream::connect(addr)?;
        stream.set_nodelay(true)?;
        let poll = Poll::new()?;
        poll.registry().register(
            &mut stream,
            Self::CLIENT,
            Interest::READABLE | Interest::WRITABLE,
        )?;
        Ok(WebSocketConnector(Some(ConnectionAttemptInternal {
            status: ConnectionStatus::TcpInProgress((stream, req.clone())),
            poll,
            events: Events::with_capacity(1024),
            addrs,
            req,
        })))
    }

    pub fn step_connect(&mut self) -> Result<Option<WebSocket>, WsError> {
        let mut current = None;
        std::mem::swap(&mut self.0, &mut current);
        let Some(mut current) = current else {
            return Err(WsError::AlreadyFailed);
        };
        current.events.clear();
        current
            .poll
            .poll(&mut current.events, Some(Duration::ZERO))?;
        for event in current.events.iter() {
            if event.is_read_closed() || event.is_write_closed() {
                return Err(WsError::Failed);
            }
            match current.status.try_extract(event) {
                Ok(Ok(wsandr)) => {
                    return Ok(Some(WebSocket {
                        socket: wsandr.0,
                        _poll: current.poll,
                    }))
                }
                Err(e) => match current.addrs.next() {
                    None => return Err(e),
                    Some(addr) => {
                        let mut stream = TcpStream::connect(addr)?;
                        stream.set_nodelay(true)?;
                        let poll = Poll::new()?;
                        poll.registry().register(
                            &mut stream,
                            Self::CLIENT,
                            Interest::READABLE | Interest::WRITABLE,
                        )?;
                        current.status =
                            ConnectionStatus::TcpInProgress((stream, current.req.clone()));
                        current.poll = poll;
                    }
                },
                Ok(Err(new_ca)) => current.status = new_ca,
            }
        }
        std::mem::swap(&mut self.0, &mut Some(current));
        Ok(None)
    }
}

type FinishedWs = (
    tungstenite::WebSocket<TcpStream>,
    tungstenite::handshake::client::Response,
);

type WaitingForWs = MidHandshake<ClientHandshake<TcpStream>>;

type WaitingForTcp = (TcpStream, http::Request<()>);

#[allow(clippy::large_enum_variant)]
enum ConnectionStatus {
    TcpInProgress(WaitingForTcp),
    WsInProgress(WaitingForWs),
    Completed(FinishedWs),
}

impl TryFrom<Result<FinishedWs, HandshakeError<ClientHandshake<TcpStream>>>> for ConnectionStatus {
    type Error = WsError;
    fn try_from(
        value: Result<FinishedWs, HandshakeError<ClientHandshake<TcpStream>>>,
    ) -> Result<Self, Self::Error> {
        use ConnectionStatus::*;
        match value {
            Ok(ws_and_r) => Ok(Completed(ws_and_r)),
            Err(HandshakeError::Interrupted(handshake)) => Ok(WsInProgress(handshake)),
            Err(HandshakeError::Failure(e)) => Err(e.into()),
        }
    }
}

impl From<ConnectionStatus> for Result<FinishedWs, ConnectionStatus> {
    fn from(val: ConnectionStatus) -> Self {
        use ConnectionStatus::*;
        match val {
            Completed(w) => Ok(w),
            WsInProgress(h) => Err(WsInProgress(h)),
            TcpInProgress(_) => unreachable!("Can not get an Tcp listener out of an ws handshake."),
        }
    }
}

impl ConnectionStatus {
    fn try_extract(self, event: &Event) -> Result<Result<FinishedWs, Self>, WsError> {
        match self {
            ConnectionStatus::Completed(w) => Ok(Ok(w)),
            ConnectionStatus::WsInProgress(handshake) => {
                ConnectionStatus::try_from(handshake.handshake()).map(ConnectionStatus::into)
            }
            ConnectionStatus::TcpInProgress((stream, req)) => {
                if !event.is_writable() {
                    return Ok(Err(ConnectionStatus::TcpInProgress((stream, req))));
                }
                let err = stream.take_error()?;
                if let Some(err) = err {
                    return Err(err.into());
                }
                match stream.peer_addr() {
                    Ok(_) => ConnectionStatus::try_from(tungstenite::client::client(req, stream))
                        .map(ConnectionStatus::into),
                    Err(err)
                        if err.kind() == io::ErrorKind::NotConnected
                            || err.raw_os_error() == Some(libc::EINPROGRESS) =>
                    {
                        Ok(Err(ConnectionStatus::TcpInProgress((stream, req))))
                    }
                    Err(err) => Err(err.into()),
                }
            }
        }
    }
}
