#[cfg(not(target_family = "wasm"))]
mod native;
#[cfg(target_family = "wasm")]
mod wasm;

/// A WebSocket implemention for native and wasm.
///
/// Example for bevy usage:
/// ```ignore
/// #[derive(Debug, Resource, Deref, DerefMut)]
/// struct CA(WebSocketConnector);
///
/// #[derive(Debug, Resource, Deref, DerefMut)]
/// struct WS(WebSocket);
///
/// fn test(mut commands: Commands, ca: Option<ResMut<CA>>, ws: Option<ResMut<WS>>) {
///     if let Some(mut ws) = ws {
///         match ws.read::<String>() {
///             Ok(Some(msg)) => {
///                 if let Err(err) = ws.write(msg) {
///                     info!("write err: {:?}", err);
///                 }
///             }
///             Err(err) => info!("read err: {:?}", err),
///             _ => (),
///         }
///         return;
///     }
///     if let Some(mut ca) = ca {
///         match ca.step_connect() {
///             Ok(Some(socket)) => {
///                 commands.insert_resource(WS(socket));
///             }
///             Err(err) => info!("err: {:?}", err),
///             Ok(_) => (),
///         }
///         return;
///     }
///     commands.insert_resource(CA(WebSocketConnector::new(
///         Uri::builder()
///             .scheme("ws")
///             .authority("localhost:8081")
///             .path_and_query("/")
///             .build()
///             .unwrap(),
///     )
///     .unwrap()));
/// }
///
/// ```
#[cfg(not(target_family = "wasm"))]
pub use native::{WebSocket, WebSocketConnector};
#[cfg(target_family = "wasm")]
pub use wasm::{WebSocket, WebSocketConnector};

#[derive(Debug)]
pub enum WsError {
    Failed,
    Closed,
    Url,
    AlreadyFailed,
    ParseError(String),
}
pub type WsMessage = String;
