/// line scroll unit
const LINE_HEIGHT: f32 = 24.;

use bevy::{
    input::mouse::{MouseScrollUnit, MouseWheel},
    picking::focus::HoverMap,
    prelude::*,
};

use crate::settings::ActiveSettingsBank;

/// Indicates that a bevy ui node may overflow
/// and requires mouse or touch scroll gestures
#[derive(Component)]
pub struct Scrollable;

/// Updates the scroll position of scrollable nodes in response to mouse input
pub fn update_scroll_position(
    mut mouse_wheel_events: EventReader<MouseWheel>,
    touches: ResMut<Touches>,
    hover_map: Res<HoverMap>,
    mut scrolled_node_query: Query<&mut ScrollPosition>,
    keyboard_input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
) {
    // scroll wheel movement
    for mouse_wheel_event in mouse_wheel_events.read() {
        let (mut dx, mut dy) = match mouse_wheel_event.unit {
            MouseScrollUnit::Line => (
                mouse_wheel_event.x * LINE_HEIGHT,
                mouse_wheel_event.y * LINE_HEIGHT,
            ),
            MouseScrollUnit::Pixel => (mouse_wheel_event.x, mouse_wheel_event.y),
        };

        if settings
            .keybindings
            .scroll_swap
            .any(|code| keyboard_input.pressed(code))
        {
            std::mem::swap(&mut dx, &mut dy);
        }

        for (_pointer, pointer_map) in hover_map.iter() {
            for (entity, _hit) in pointer_map.iter() {
                if let Ok(mut scroll_position) = scrolled_node_query.get_mut(*entity) {
                    scroll_position.offset_x -= dx;
                    scroll_position.offset_y -= dy;
                }
            }
        }
    }
    // touch scroll movement
    for (pointer, pointer_map) in hover_map.iter() {
        if let Some(touch) = pointer
            .get_touch_id()
            .and_then(|tid| touches.get_pressed(tid))
        {
            for (entity, _hit) in pointer_map.iter() {
                if let Ok(mut scroll_position) = scrolled_node_query.get_mut(*entity) {
                    scroll_position.offset_x -= touch.delta().x;
                    scroll_position.offset_y -= touch.delta().y;
                }
            }
        }
    }
}
