use bevy::app::Plugin;
use bevy::prelude::*;

use crate::game::gamemodes::campaign::levels::CampaignGroupWrapper;
use crate::game::gamemodes::campaign::{CampaignModeConfig, LEVEL_TRANSLATION_ID_BASE};
use crate::settings::{ActiveSettingsBank, PersistentSettingsBank};
use crate::ui::BackEvent;

use super::selector::{Selector, SelectorOption, SelectorSelected};
use super::SetupMenuState;

pub struct SetupCampaignPlugin;

impl Plugin for SetupCampaignPlugin {
    fn build(&self, app: &mut App) {
        // group selector
        app.add_systems(
            OnEnter(SetupMenuState::CampaignGroupSelect),
            spawn_group_layout,
        );
        app.add_systems(
            Update,
            handle_group_select
                .run_if(in_state(SetupMenuState::CampaignGroupSelect))
                .run_if(on_event::<SelectorSelected>),
        );
        app.add_systems(
            Update,
            handle_group_back
                .run_if(in_state(SetupMenuState::CampaignGroupSelect))
                .run_if(on_event::<BackEvent>),
        );
        // level selector
        app.add_systems(
            OnEnter(SetupMenuState::CampaignLevelSelect),
            spawn_level_layout,
        );
        app.add_systems(
            Update,
            handle_level_select
                .run_if(in_state(SetupMenuState::CampaignLevelSelect))
                .run_if(on_event::<SelectorSelected>),
        );
        app.add_systems(
            Update,
            handle_level_back
                .run_if(in_state(SetupMenuState::CampaignLevelSelect))
                .run_if(on_event::<BackEvent>),
        );
    }
}

/* GROUP */

fn spawn_group_layout(mut commands: Commands, campaign_config: Res<CampaignModeConfig>) {
    let source = CampaignGroupWrapper::list_ids();
    let selected = source
        .iter()
        .position(|group_id| campaign_config.level.get_group_id() == *group_id)
        .unwrap_or(0);
    let options: Vec<SelectorOption> = source
        .iter()
        .enumerate()
        .map(|(index, group_id)| SelectorOption {
            name: format!("{}-{}", LEVEL_TRANSLATION_ID_BASE, group_id),
            description: Some(format!(
                "{}-{}-description",
                LEVEL_TRANSLATION_ID_BASE, group_id
            )),
            icon: None,
            index,
        })
        .collect();

    let scope = SetupMenuState::CampaignGroupSelect;
    commands.spawn((
        Selector {
            title: "setup-menu-group".to_string(),
            selected,
            options,
        },
        StateScoped(scope),
    ));
}

fn handle_group_select(
    selectors: Query<(Entity, &Selector)>,
    mut setup_state: ResMut<NextState<SetupMenuState>>,
    mut settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut campaign_config: ResMut<CampaignModeConfig>,
    mut selected: EventReader<SelectorSelected>,
) {
    let Ok(selector) = selectors.get_single() else {
        warn!("Only one selector should be active at any point");
        return;
    };
    if selected.is_empty() {
        return;
    }
    selected.clear();

    let groups = CampaignGroupWrapper::list_ids();
    let group_id = groups
        .get(selector.1.selected)
        .expect("The selector is created from the available groups");
    let group_default = CampaignGroupWrapper::from_group_default(group_id)
        .expect("The group id was taken from the group id list, so it has to exist");

    // only override the group if the last selected level is from a different group
    if *group_id != campaign_config.level.get_group_id() {
        campaign_config.level = group_default;
        settings.game_behaviour.campaign_level = campaign_config.level.clone();
        settings_bank_a.game_behaviour.campaign_level = campaign_config.level.clone();
    }

    setup_state.set(SetupMenuState::CampaignLevelSelect);
}

fn handle_group_back(mut setup_state: ResMut<NextState<SetupMenuState>>) {
    setup_state.set(SetupMenuState::GameModeSelect);
}

/* LEVEL */

fn spawn_level_layout(mut commands: Commands, campaign_config: Res<CampaignModeConfig>) {
    let group_id = campaign_config.level.get_group_id();
    let source = CampaignGroupWrapper::list_level_ids(group_id).expect(
        "The group id was taken from a valid CampaignGroupWrapper, so the group has to exist",
    );
    let selected = source
        .iter()
        .position(|level_id| campaign_config.level.get_level_id() == *level_id)
        .unwrap_or(0);
    let options: Vec<SelectorOption> = source
        .iter()
        .enumerate()
        .map(|(index, level_id)| SelectorOption {
            name: format!("{}-{}-{}", LEVEL_TRANSLATION_ID_BASE, group_id, level_id),
            description: Some(format!(
                "{}-{}-{}-description",
                LEVEL_TRANSLATION_ID_BASE, group_id, level_id
            )),
            icon: None,
            index,
        })
        .collect();

    let scope = SetupMenuState::CampaignLevelSelect;
    commands.spawn((
        Selector {
            title: "setup-menu-level".to_string(),
            selected,
            options,
        },
        StateScoped(scope),
    ));
}

fn handle_level_select(
    selectors: Query<(Entity, &Selector)>,
    mut setup_state: ResMut<NextState<SetupMenuState>>,
    mut settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut campaign_config: ResMut<CampaignModeConfig>,
    mut selected: EventReader<SelectorSelected>,
) {
    let Ok(selector) = selectors.get_single() else {
        warn!("Only one selector should be active at any point");
        return;
    };
    if selected.is_empty() {
        return;
    }
    selected.clear();

    let group_id = campaign_config.level.get_group_id();
    let levels = CampaignGroupWrapper::list_level_ids(group_id).expect(
        "The group id was taken from a valid CampaignGroupWrapper, so the group has to exist",
    );
    let level_id = levels
        .get(selector.1.selected)
        .expect("The selector was created from the available levels");
    let level = CampaignGroupWrapper::from_ids(group_id, level_id)
        .expect("The level was taken from the groups level list, so it has to exist");

    campaign_config.level = level;
    settings.game_behaviour.campaign_level = campaign_config.level.clone();
    settings_bank_a.game_behaviour.campaign_level = campaign_config.level.clone();

    setup_state.set(SetupMenuState::ConfigureGame);
}

fn handle_level_back(mut setup_state: ResMut<NextState<SetupMenuState>>) {
    setup_state.set(SetupMenuState::CampaignGroupSelect);
}
