use crate::{
    define_asset_collection,
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        font_loader::AppFonts,
        theme::{BUTTON_COLOR, BUTTON_TEXT_COLOR, INV_SLOT_RADIUS, TITLE_TEXT_COLOR},
        utils::Scrollable,
        BackEvent, UiClickEvent,
    },
    AppMenuState,
};
use accesskit::{Node as Accessible, Role};
use bevy::{a11y::AccessibilityNode, prelude::*};

pub struct SelectorPlugin;

impl Plugin for SelectorPlugin {
    fn build(&self, app: &mut App) {
        app.add_event::<SelectorSelect>();
        app.add_event::<SelectorSelected>();

        app.add_systems(
            Update,
            spawn_layout.run_if(in_state(AppMenuState::SetupMenu)),
        );
        app.add_systems(
            Update,
            (
                process_keyboard_input,
                handle_selection_change,
                hover_feedback,
            )
                .run_if(any_with_component::<Selector>),
        );

        app.register_asset_collection::<SelectorNavigationButtons>();
    }
}

/// gap between title, gamemode cards and back button
const ROW_GAP: f32 = 1.0;
/// font size of the screen title
const TITLE_FONT_SIZE: f32 = 48.0;

/// width of the icon
const CARD_ICON_SIZE: f32 = 1.0;
/// font size of the gamemode name
const CARD_TITLE_SIZE: f32 = 28.0;
/// opacity of title card background
const CARD_TITLE_OPACITY: f32 = 0.95;
const CARD_TITLE_OPACITY_HOVER: f32 = 0.97;
/// font size of the gamemode description
const CARD_DESC_SIZE: f32 = 16.0;
/// opacity of description card background
const CARD_DESC_OPACITY: f32 = 0.9;
const CARD_DESC_OPACITY_HOVER: f32 = 0.92;
const CARD_ROW_GAP: f32 = 8.0;

const ARROW_OPACITY: f32 = 0.6;
const ARROW_OPACITY_HOVER: f32 = 0.66;

/// font size of back button
const BACK_FONT_SIZE: f32 = 24.0;
/// padding used for back button (% of width, height)
const BACK_BUTTON_PADDING: (f32, f32) = (6.0, 1.0);

#[derive(Component, Clone)]
/// Option selectable using the selector
pub struct SelectorOption {
    /// Translation ID of the option name
    pub name: String,
    /// optional icon handle, displayed left of the option name
    pub icon: Option<Handle<Image>>,
    /// optional option description, displayed below the title
    /// in a separate card
    pub description: Option<String>,
    /// internal use index
    pub index: usize,
}

#[derive(Component)]
/// Configuration data for the Selector screen
/// After you spawn this Component, the layout will be inserted into it
pub struct Selector {
    /// Translation ID of the title text
    pub title: String,
    /// selector options
    pub options: Vec<SelectorOption>,
    /// index of currently selected option
    /// assumed to exist within the options list
    pub selected: usize,
}

/// Event sent when the option is selected
#[derive(Event)]
pub struct SelectorSelected;

define_asset_collection! (
    SelectorNavigationButtons,
    !left : Image = "icons/buttons/left.png",
    !right : Image = "icons/buttons/right.png",
);

/// Marker struct for the option title card
#[derive(Component)]
struct SelectorOptionTitleCard;
/// Marker struct for the option description card
#[derive(Component)]
struct SelectorOptionDescriptionCard;

/// helper function to spawn the selector layout
/// should be called from a custom system, which assembles the options beforehand
fn spawn_layout(
    selectors: Query<(Entity, &Selector), Added<Selector>>,
    localization: Res<Localization>,
    mut commands: Commands,
    fonts: Res<AppFonts>,
    buttons: Res<SelectorNavigationButtons>,
) {
    let title_font = fonts.get_title_font();
    let paragraph_font = fonts.get_paragraph_font();

    for (entity, config) in &selectors {
        commands
            .entity(entity)
            .insert((
                Node {
                    width: Val::Percent(100.),
                    height: Val::Percent(100.),
                    flex_direction: FlexDirection::Column,
                    align_items: AlignItems::Center,
                    justify_content: JustifyContent::Center,
                    row_gap: Val::Percent(ROW_GAP),
                    ..Default::default()
                },
                PickingBehavior::IGNORE,
            ))
            .with_children(|builder| {
                builder.spawn((
                    Text::new(localization.translate(&config.title)),
                    TextFont {
                        font: title_font.clone(),
                        font_size: TITLE_FONT_SIZE,
                        ..default()
                    },
                    TextColor(TITLE_TEXT_COLOR.cast()),
                    TextLayout::default()
                        .with_justify(JustifyText::Center)
                        .with_linebreak(LineBreak::WordOrCharacter),
                ));

                let card_padding = CARD_TITLE_SIZE / 2.0;
                builder
                    .spawn((Node {
                        display: Display::Flex,
                        flex_direction: FlexDirection::Row,
                        width: Val::Auto,
                        height: Val::Auto,
                        max_width: Val::Vw(100.0),
                        // margin towards the edge corners
                        padding: UiRect::horizontal(Val::Px(card_padding)),
                        column_gap: Val::Percent(ROW_GAP),
                        align_items: AlignItems::Center,
                        ..default()
                    },))
                    .insert(PickingBehavior {
                        should_block_lower: false,
                        ..default()
                    })
                    .with_children(|builder| {
                        // left button
                        builder.spawn((
                            ImageNode {
                                image: buttons.left.clone_weak(),
                                color: BUTTON_COLOR.cast().with_alpha(ARROW_OPACITY),
                                ..default()
                            },
                            Node {
                                // ensure the icon is square
                                aspect_ratio: Some(1.0),
                                height: Val::Px(CARD_TITLE_SIZE * CARD_ICON_SIZE),
                                ..default()
                            },
                            Button,
                        )).observe(move |
                                   trigger: Trigger<Pointer<Click>>,
                                   mut click_event: EventWriter<UiClickEvent>,
                                   mut change: EventWriter<SelectorSelect>
                                   | {
                                       if trigger.event().button == PointerButton::Primary {
                                           click_event.send(UiClickEvent);
                                           change.send(SelectorSelect::Prev);
                                       }
                                   });

                        builder
                            .spawn((Node {
                                display: Display::Grid,
                                grid_template_columns: RepeatedGridTrack::min_content(1),
                                grid_template_rows: RepeatedGridTrack::min_content(1),
                                width: Val::Auto,
                                height: Val::Auto,
                                max_width: Val::Vw(100.0),
                                column_gap: Val::Percent(ROW_GAP),
                                align_items: AlignItems::Center,
                                ..default()
                            },))
                            .insert(PickingBehavior {
                                should_block_lower: false,
                                ..default()
                            })
                            .with_children(|builder| {
                                // selectable options
                                for option in &config.options {
                                    let name = localization.translate(&option.name);
                                    let icon = option.icon.clone();

                                    builder
                                .spawn((
                                    Node {
                                        display:  Display::Flex,
                                        width: Val::Px(660.0),
                                        max_width: Val::Vw(80.0),
                                        flex_direction: FlexDirection::Column,
                                        justify_content: JustifyContent::Center,
                                        row_gap: Val::Percent(CARD_ROW_GAP),
                                        grid_column: GridPlacement::start(1),
                                        grid_row: GridPlacement::start(1),
                                        ..default()
                                    },
                                    if option.index == config.selected {
                                        Visibility::Visible
                                    } else {
                                        Visibility::Hidden
                                    },
                                    option.clone(),
                                    Button,
                                ))
                                .observe(
                                    move |trigger: Trigger<Pointer<Click>>,
                                    mut click_event: EventWriter<UiClickEvent>,
                                    mut selected: EventWriter<SelectorSelected>| {
                                        if trigger.event().button == PointerButton::Primary {
                                            selected.send(SelectorSelected);
                                            click_event.send(UiClickEvent);
                                        }
                                    },
                                )
                                .with_children(|builder| {
                                    builder
                                        .spawn((
                                            Node {
                                                display: Display::Flex,
                                                flex_direction: FlexDirection::Row,
                                                justify_content: JustifyContent::Center,
                                                align_items: AlignItems::Center,
                                                padding: UiRect::all(Val::Px(card_padding)),
                                                column_gap: Val::Percent(4.0),
                                                ..Default::default()
                                            },
                                            AccessibilityNode(Accessible::new(Role::ListItem)),
                                            BackgroundColor(
                                                BUTTON_COLOR.cast().with_alpha(CARD_TITLE_OPACITY),
                                            ),
                                            BorderRadius::all(Val::Px(INV_SLOT_RADIUS * 2.0)),
                                            SelectorOptionTitleCard,
                                        ))
                                        .with_children(|builder| {
                                            if let Some(icon) = icon {
                                                builder
                                                    .spawn((
                                                        ImageNode {
                                                            image: icon,
                                                            color: BUTTON_TEXT_COLOR.cast(),
                                                            ..default()
                                                        },
                                                        Node {
                                                            // ensure the icon is square
                                                            aspect_ratio: Some(1.0),
                                                            height: Val::Px(
                                                                CARD_TITLE_SIZE
                                                                    * CARD_ICON_SIZE,
                                                            ),
                                                            ..default()
                                                        },
                                                    ));
                                            }
                                            builder
                                                .spawn((
                                                    Text::new(name),
                                                    TextFont {
                                                        font: title_font.clone(),
                                                        font_size: CARD_TITLE_SIZE,
                                                        ..default()
                                                    },
                                                    TextColor(BUTTON_TEXT_COLOR.cast()),
                                                    TextLayout::default()
                                                        .with_linebreak(LineBreak::WordOrCharacter),
                                                    Label,
                                                ));
                                        });

                                    if let Some(description) = &option.description {
                                        let description = localization.translate(description);
                                        builder
                                            .spawn((
                                                Node {
                                                    display: Display::Flex,
                                                    align_items: AlignItems::Center,
                                                    justify_content: JustifyContent::Center,
                                                    overflow: Overflow::scroll_y(),
                                                    overflow_clip_margin:
                                                        OverflowClipMargin::padding_box(),
                                                    padding: UiRect::all(Val::Px(card_padding)),
                                                    ..Default::default()
                                                },
                                                Scrollable,
                                                AccessibilityNode(Accessible::new(Role::ListItem)),
                                                BackgroundColor(
                                                    BUTTON_COLOR
                                                        .cast()
                                                        .with_alpha(CARD_DESC_OPACITY),
                                                ),
                                                BorderRadius::all(Val::Px(INV_SLOT_RADIUS * 2.0)),
                                                SelectorOptionDescriptionCard,
                                            ))
                                            .with_children(|builder| {
                                                builder
                                                    .spawn((
                                                        Text::new(description),
                                                        TextFont {
                                                            font: paragraph_font.clone(),
                                                            font_size: CARD_DESC_SIZE,
                                                            ..default()
                                                        },
                                                        TextColor(BUTTON_TEXT_COLOR.cast()),
                                                        TextLayout::default().with_linebreak(
                                                            LineBreak::WordOrCharacter,
                                                        ),
                                                        Label,
                                                    ));
                                            });
                                    }
                                });
                                }
                            });

                        // right button
                        builder.spawn((
                            ImageNode {
                                image: buttons.right.clone_weak(),
                                color: BUTTON_COLOR.cast().with_alpha(ARROW_OPACITY),
                                ..default()
                            },
                            Node {
                                // ensure the icon is square
                                aspect_ratio: Some(1.0),
                                height: Val::Px(CARD_TITLE_SIZE * CARD_ICON_SIZE),
                                ..default()
                            },
                            Button,
                        )).observe(move |
                                   trigger: Trigger<Pointer<Click>>,
                                   mut click_event: EventWriter<UiClickEvent>,
                                   mut change: EventWriter<SelectorSelect>
                                   | {
                                       if trigger.event().button == PointerButton::Primary {
                                           click_event.send(UiClickEvent);
                                           change.send(SelectorSelect::Next);
                                       }
                                   });
                    });

                builder
                    .spawn((
                        Button,
                        BackgroundColor(BUTTON_COLOR.cast()),
                        Node {
                            width: Val::Auto,
                            height: Val::Auto,
                            flex_direction: FlexDirection::Column,
                            align_items: AlignItems::Center,
                            padding: UiRect::axes(
                                Val::Percent(BACK_BUTTON_PADDING.0),
                                Val::Percent(BACK_BUTTON_PADDING.1),
                            ),
                            ..Default::default()
                        },
                        AccessibilityNode(Accessible::new(Role::ListItem)),
                    ))
                    .observe(
                        move |trigger: Trigger<Pointer<Click>>,
                              mut click_event: EventWriter<UiClickEvent>,
                              mut back: EventWriter<BackEvent>| {
                            if trigger.event().button == PointerButton::Primary {
                                click_event.send(UiClickEvent);
                                back.send(BackEvent);
                            }
                        },
                    )
                    .with_children(|builder| {
                        builder.spawn((
                            Text::new(localization.translate("back")),
                            TextFont {
                                font: title_font.clone(),
                                font_size: BACK_FONT_SIZE,
                                ..default()
                            },
                            TextColor(BUTTON_TEXT_COLOR.cast()),
                        ));
                    });
            });
    }
}

#[derive(Event)]
enum SelectorSelect {
    Prev,
    Next,
}
fn handle_selection_change(
    mut selectors: Query<(Entity, &mut Selector)>,
    mut options: Query<(Entity, &mut Visibility, &SelectorOption)>,
    mut events: EventReader<SelectorSelect>,
) {
    let Ok(mut selector) = selectors.get_single_mut() else {
        warn!("Only one selector should be active at any point");
        return;
    };
    for ev in events.read() {
        let current = selector.1.selected;
        let option_count = selector.1.options.len();
        match ev {
            SelectorSelect::Prev => {
                let new = if current == 0 {
                    option_count - 1
                } else {
                    current - 1
                };
                selector.1.selected = new;
                for (_, mut vis, option) in options.iter_mut() {
                    *vis = if option.index == new {
                        Visibility::Visible
                    } else {
                        Visibility::Hidden
                    };
                }
            }
            SelectorSelect::Next => {
                let new = if current + 1 == option_count {
                    0
                } else {
                    current + 1
                };
                selector.1.selected = new;
                for (_, mut vis, option) in options.iter_mut() {
                    *vis = if option.index == new {
                        Visibility::Visible
                    } else {
                        Visibility::Hidden
                    };
                }
            }
        }
    }
}

fn process_keyboard_input(
    input: Res<ButtonInput<KeyCode>>,
    mut select: EventWriter<SelectorSelect>,
    mut click_event: EventWriter<UiClickEvent>,
    settings: ResMut<ActiveSettingsBank>,
    mut selected: EventWriter<SelectorSelected>,
    mut back: EventWriter<BackEvent>,
) {
    // previous
    if settings
        .keybindings
        .menu_prev
        .any(|code| input.just_pressed(code))
    {
        select.send(SelectorSelect::Prev);
        click_event.send(UiClickEvent);
    }
    // next
    if settings
        .keybindings
        .menu_next
        .any(|code| input.just_pressed(code))
    {
        select.send(SelectorSelect::Next);
        click_event.send(UiClickEvent);
    }
    // select
    if settings
        .keybindings
        .menu_select
        .any(|code| input.just_pressed(code))
    {
        selected.send(SelectorSelected);
        click_event.send(UiClickEvent);
    }
    // back
    if settings
        .keybindings
        .menu_back
        .any(|code| input.just_pressed(code))
    {
        back.send(BackEvent);
        click_event.send(UiClickEvent);
    }
}

#[allow(clippy::type_complexity)]
fn hover_feedback(
    // main card hover
    option_interactions: Query<
        (&Interaction, &Children),
        (Changed<Interaction>, With<SelectorOption>),
    >,
    mut title_cards: Query<
        &mut BackgroundColor,
        (
            With<SelectorOptionTitleCard>,
            Without<SelectorOptionDescriptionCard>,
        ),
    >,
    mut desc_cards: Query<
        &mut BackgroundColor,
        (
            With<SelectorOptionDescriptionCard>,
            Without<SelectorOptionTitleCard>,
        ),
    >,
    // left/right arrow button
    mut arrow_interactions: Query<(&Interaction, &mut ImageNode)>,
) {
    for (interaction, children) in option_interactions.iter() {
        match interaction {
            Interaction::Pressed | Interaction::Hovered => {
                // NOTE: both cards are hovered at the same time,
                // because they trigger the same action
                title_cards
                    .get_mut(children[0])
                    .expect("The layout system creates two children with the appropriate markers")
                    .0
                    .set_alpha(CARD_TITLE_OPACITY_HOVER);
                if let Some(entity) = children.get(1) {
                    desc_cards
                        .get_mut(*entity)
                        .expect(
                            "The layout system creates two children with the appropriate markers",
                        )
                        .0
                        .set_alpha(CARD_DESC_OPACITY_HOVER);
                }
            }
            Interaction::None => {
                title_cards
                    .get_mut(children[0])
                    .expect("The layout system creates two children with the appropriate markers")
                    .0
                    .set_alpha(CARD_TITLE_OPACITY);
                if let Some(entity) = children.get(1) {
                    desc_cards
                        .get_mut(*entity)
                        .expect(
                            "The layout system creates two children with the appropriate markers",
                        )
                        .0
                        .set_alpha(CARD_DESC_OPACITY);
                }
            }
        }
    }

    for (interaction, mut img) in arrow_interactions.iter_mut() {
        match interaction {
            Interaction::Pressed | Interaction::Hovered => img.color.set_alpha(ARROW_OPACITY_HOVER),
            Interaction::None => img.color.set_alpha(ARROW_OPACITY),
        }
    }
}
