use crate::{
    define_asset_collection,
    game::{asset_loading::LACManager, gamemodes::GameMode},
    prelude::*,
    settings::{ActiveSettingsBank, PersistentSettingsBank},
    ui::{setup_menu::selector::Selector, BackEvent},
    AppMenuState, GameConfig,
};
use bevy::prelude::*;

use super::{
    selector::{SelectorOption, SelectorSelected},
    SetupMenuState,
};

#[derive(Resource, Default)]
pub struct GameModeSelectPlugin;
impl Plugin for GameModeSelectPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<GameModeIcons>();

        app.add_systems(OnEnter(SetupMenuState::GameModeSelect), spawn_layout);
        app.add_systems(
            Update,
            handle_select
                .run_if(in_state(SetupMenuState::GameModeSelect))
                .run_if(on_event::<SelectorSelected>),
        );

        app.add_systems(
            Update,
            handle_back
                .run_if(in_state(SetupMenuState::GameModeSelect))
                .run_if(on_event::<BackEvent>),
        );
    }
}

define_asset_collection!(
    GameModeIcons,
    !challenge : Image = "icons/gamemodes/challenge.png",
    !creative : Image = "icons/gamemodes/creative.png",
    !tutorial : Image = "icons/gamemodes/tutorial.png",
    !zen : Image = "icons/gamemodes/zen.png",
    err : "gamemode-icon-missing" "Gamemode icon missing." "gamemode-icon-missing-desc" "The gamemode selection screen required the gamemode icons"
);

impl GameModeIcons {
    fn get_gamemode_icon(&self, mode: &GameMode) -> Handle<Image> {
        match mode {
            GameMode::Creative => self.creative.clone_weak(),
            GameMode::Challenge => self.challenge.clone_weak(),
            GameMode::Campaign => self.tutorial.clone_weak(),
            GameMode::Zen => self.zen.clone_weak(),
        }
    }
}

fn spawn_layout(mut commands: Commands, icons: Res<GameModeIcons>, config: Res<GameConfig>) {
    let source = GameMode::list();
    let selected = source
        .iter()
        .position(|g| *g == config.selected_gamemode)
        .unwrap_or(0);
    let options: Vec<SelectorOption> = source
        .iter()
        .enumerate()
        .map(|(index, gamemode)| SelectorOption {
            name: gamemode.get_translation_id().to_string(),
            description: Some(format!("{}-description", gamemode.get_translation_id())),
            icon: Some(icons.get_gamemode_icon(gamemode)),
            index,
        })
        .collect();

    let scope = SetupMenuState::GameModeSelect;
    commands.spawn((
        Selector {
            title: "setup_menu-select_gamemode".to_string(),
            selected,
            options,
        },
        StateScoped(scope),
    ));
}

fn handle_select(
    selectors: Query<(Entity, &Selector)>,
    mut setup_state: ResMut<NextState<SetupMenuState>>,
    mut settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut config: ResMut<GameConfig>,
    mut selected: EventReader<SelectorSelected>,
) {
    let Ok(selector) = selectors.get_single() else {
        warn!("Only one selector should be active at any point");
        return;
    };
    if selected.is_empty() {
        return;
    }
    selected.clear();
    let gamemodes = GameMode::list();
    let gamemode = gamemodes
        .get(selector.1.selected)
        .expect("The selector is created from the gamemode list");
    config.selected_gamemode = gamemode.clone();
    settings.game_behaviour.gamemode = gamemode.clone();
    settings_bank_a.game_behaviour.gamemode = gamemode.clone();
    let next = match gamemode {
        GameMode::Campaign => SetupMenuState::CampaignGroupSelect,
        _ => SetupMenuState::ConfigureGame,
    };
    setup_state.set(next);
}

fn handle_back(mut app_state: ResMut<NextState<AppMenuState>>) {
    app_state.set(AppMenuState::MainMenu);
}
