use crate::{
    game::{
        bot::{
            challenge_bot::start_challenge_bot_game, creative_bot::start_creative_bot_game,
            BotConnectingState,
        },
        gamemodes::{
            campaign::{levels::init_campaign_mode, CampaignModeConfig},
            challenge::{init_challenge_mode, ChallengeModeConfig},
            creative::{init_creative_mode, CreativeModeConfig},
            zen::{init_zen_mode, ZenModeConfig},
            GameMode,
        },
    },
    i18n::Localization,
    prelude::*,
    settings::{ActiveSettingsBank, PersistentSettingsBank},
    ui::{
        is_typing_egui,
        theme::{self, *},
        BackEvent, UiClickEvent,
    },
    AppMenuState, GameConfig,
};
use bevy::{ecs::system::RunSystemOnce, prelude::*};
use bevy_egui::{
    egui::{self, Align2, FontId, RichText},
    EguiContexts,
};

use super::SetupMenuState;

#[derive(Resource, Default)]
pub struct ConfigureGamePlugin;
impl Plugin for ConfigureGamePlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (configure_game_screen, process_keyboard_input)
                .run_if(in_state(SetupMenuState::ConfigureGame)),
        );
        app.add_systems(
            Update,
            handle_back
                .run_if(in_state(SetupMenuState::ConfigureGame))
                .run_if(on_event::<BackEvent>),
        );
        app.add_systems(
            OnEnter(AppMenuState::SetupMenu),
            (generate_new_seed, refill_memory).chain(),
        );
    }
}

/// generates a new random seed
pub fn generate_new_seed(
    config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut zen_config: ResMut<ZenModeConfig>,
) {
    if !config.skip_seed {
        challenge_config.seed = fastrand::u64(..).to_string();
        zen_config.seed.clone_from(&challenge_config.seed);
    }
}

/// prefills the `GameConfigMemory` with data from the game_behaviour settings
pub fn refill_memory(
    mut config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut campaign_config: ResMut<CampaignModeConfig>,
    settings: Res<ActiveSettingsBank>,
) {
    if !config.skip_mode {
        challenge_config.difficulty = settings.game_behaviour.difficulty.clone();
        campaign_config.level = settings.game_behaviour.campaign_level.clone();
        config.selected_gamemode = settings.game_behaviour.gamemode.clone();
    }
    if !config.skip_bot {
        config.bot_url.clone_from(&settings.game_behaviour.bot_url);
        config.use_bot = settings.game_behaviour.bot_usage;
    }
    if !config.skip_recorder {
        config
            .replay_name
            .clone_from(&settings.game_behaviour.replay_name);
        config.activate_recorder = settings.game_behaviour.activate_recorder;
    }
}

#[allow(clippy::too_many_arguments)]
pub fn configure_game_screen(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut settings: ResMut<ActiveSettingsBank>,
    settings_bank_a: ResMut<PersistentSettingsBank>,
    mut config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut zen_config: ResMut<ZenModeConfig>,
    campaign_config: Res<CampaignModeConfig>,
    mut _creative_config: ResMut<CreativeModeConfig>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
    commands: Commands,
    mut bot_state: ResMut<BotConnectingState>,
    mut back_event: EventWriter<BackEvent>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    egui::Area::new("setup_settings_area".into())
        .anchor(Align2::CENTER_CENTER, [0., 0.])
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;

            ui.label(
                RichText::new(
                    localization.translate(config.selected_gamemode.get_translation_id()),
                )
                .font(FontId::new(
                    window.y * BIG_TEXT_SIZE,
                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                ))
                .color(TITLE_TEXT_COLOR),
            );
            ui.add_space(window.y * SPACE_SIZE);

            if *bot_state == BotConnectingState::None {
                // setting up
                match config.selected_gamemode {
                    GameMode::Challenge => {
                        ui.label(
                            RichText::new(localization.translate("setup-menu-seed"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );

                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut challenge_config.seed)
                                .min_size(egui::Vec2::new(window.x * 0.5, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.add_space(window.y * SPACE_SIZE);

                        /* ui.columns(2, |columns| {
                            columns[0].label(
                                RichText::new(localization.translate("setup-menu-difficulty"))
                                    .font(FontId::new(
                                        window.y * SMALL_TEXT_SIZE,
                                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                    ))
                                    .color(NORMAL_TEXT_COLOR),
                            );
                            egui::ComboBox::from_id_source("difficulty")
                                .selected_text(challenge_config.difficulty.get_name(&localization))
                                .show_ui(&mut columns[1], |ui| {
                                    for dif in Difficulty::list() {
                                        let difficulty_name = dif.get_name(&localization);
                                        if ui
                                            .selectable_value(
                                                &mut challenge_config.difficulty,
                                                dif,
                                                difficulty_name,
                                            )
                                            .clicked()
                                        {
                                            settings.game_behaviour.difficulty =
                                                challenge_config.difficulty.clone();
                                            settings_bank_a.game_behaviour.difficulty =
                                                settings.game_behaviour.difficulty.clone();
                                            click_event.send(UiClickEvent);
                                        };
                                    }
                                });
                        }); */
                    }
                    GameMode::Creative => (),
                    GameMode::Campaign => (),
                    GameMode::Zen => {
                        ui.label(
                            RichText::new(localization.translate("setup-menu-seed"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );

                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut zen_config.seed)
                                .min_size(egui::Vec2::new(window.x * 0.5, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.add_space(window.y * SPACE_SIZE);
                    }
                }

                ui.add_space(window.y * SPACE_SIZE * 2.0);

                let allow_bot_connections = match config.selected_gamemode {
                    // allow connectign bot in creative and challenge mode
                    GameMode::Challenge | GameMode::Creative | GameMode::Zen => true,
                    // only allow bot connections in supported campaign levels
                    GameMode::Campaign => campaign_config.level.get_bot_type().is_some(),
                };
                if allow_bot_connections {
                    ui.horizontal(|ui| {
                        ui.label(
                            RichText::new(localization.translate("bot-url"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );
                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut config.bot_url)
                                .min_size(egui::Vec2::new(window.x * 0.25, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.style_mut().visuals.override_text_color = Some(NORMAL_TEXT_COLOR);
                        ui.checkbox(&mut config.use_bot, localization.translate("bot-use"));
                        ui.style_mut().visuals.override_text_color = None;
                    });

                    ui.add_space(window.y * SPACE_SIZE * 2.0);
                }

                // game recorder
                if config.selected_gamemode == GameMode::Challenge
                    || config.selected_gamemode == GameMode::Zen
                {
                    ui.horizontal(|ui| {
                        ui.label(
                            RichText::new(localization.translate("game-record-name"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );
                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut config.replay_name)
                                .min_size(egui::Vec2::new(window.x * 0.2, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.style_mut().visuals.override_text_color = Some(NORMAL_TEXT_COLOR);
                        ui.checkbox(
                            &mut config.activate_recorder,
                            localization.translate("activate-recorder"),
                        );
                        ui.style_mut().visuals.override_text_color = None;
                    });
                    ui.add_space(window.y * SPACE_SIZE * 2.0);
                }

                ui.horizontal_centered(|ui| {
                    ui.style_mut().text_styles.insert(
                        egui::TextStyle::Button,
                        FontId::new(
                            window.y * SECONDARY_BUTTON_TEXT_SIZE,
                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                        ),
                    );
                    if ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("back"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        back_event.send(BackEvent);
                        click_event.send(UiClickEvent);
                    }
                    if ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("start-game"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        start_game_internal(
                            &mut settings,
                            settings_bank_a,
                            &config,
                            &mut click_event,
                            commands,
                            allow_bot_connections,
                            &campaign_config,
                        );
                    }
                });
            } else {
                // connecting or failed
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("bot-state"),
                        localization.translate(bot_state.get_translation_id())
                    ))
                    .font(FontId::new(
                        window.y * SMALL_TEXT_SIZE,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ))
                    .color(NORMAL_TEXT_COLOR),
                );
                if !bot_state.is_waiting_state()
                    && ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("back"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                {
                    *bot_state = BotConnectingState::None;
                }
            }
        });
}

#[allow(clippy::too_many_arguments)]
fn start_game_internal(
    settings: &mut ActiveSettingsBank,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    config: &GameConfig,
    click_event: &mut EventWriter<UiClickEvent>,
    mut commands: Commands,
    allow_bot_connections: bool,
    campaign_config: &CampaignModeConfig,
) {
    click_event.send(UiClickEvent);
    settings.game_behaviour.bot_usage = config.use_bot;
    settings_bank_a.game_behaviour.bot_usage = config.use_bot;
    settings.game_behaviour.bot_url.clone_from(&config.bot_url);
    settings_bank_a
        .game_behaviour
        .bot_url
        .clone_from(&config.bot_url);

    settings.game_behaviour.activate_recorder = config.activate_recorder;
    settings_bank_a.game_behaviour.activate_recorder = config.activate_recorder;

    settings
        .game_behaviour
        .replay_name
        .clone_from(&config.replay_name);
    settings_bank_a
        .game_behaviour
        .replay_name
        .clone_from(&config.replay_name);

    let mode = config.selected_gamemode.clone();
    if config.use_bot && allow_bot_connections {
        let campaign_level = campaign_config.level.clone();
        commands.queue(move |w: &mut World| {
            match mode {
                GameMode::Challenge => {
                    w.run_system_once(start_challenge_bot_game).unwrap();
                }
                GameMode::Zen => {
                    w.run_system_once(start_challenge_bot_game).unwrap();
                }
                GameMode::Creative => {
                    w.run_system_once(start_creative_bot_game).unwrap();
                }
                GameMode::Campaign => {
                    // try to initialize a campaign bot for the given level
                    if !campaign_level.init_bot(w) {
                        // the initialization returns false if the level does not support bots
                        // in this case we have to fall back to the normal campaign init method,
                        // as we might still have the old bot enabled state for compatible levels
                        // NOTE: should never happen, as this block only runs when bot connections are allowed
                        w.run_system_once(init_campaign_mode).unwrap();
                    }
                }
            };
        });
    } else {
        commands.queue(move |w: &mut World| {
            match mode {
                GameMode::Challenge => w.run_system_once(init_challenge_mode).unwrap(),
                GameMode::Creative => w.run_system_once(init_creative_mode).unwrap(),
                GameMode::Campaign => w.run_system_once(init_campaign_mode).unwrap(),
                GameMode::Zen => w.run_system_once(init_zen_mode).unwrap(),
            };
        });
    }
}

#[allow(clippy::too_many_arguments)]
fn process_keyboard_input(
    mut contexts: EguiContexts,
    input: Res<ButtonInput<KeyCode>>,
    mut click_event: EventWriter<UiClickEvent>,
    mut settings: ResMut<ActiveSettingsBank>,
    settings_bank_a: ResMut<PersistentSettingsBank>,
    commands: Commands,
    config: Res<GameConfig>,
    campaign_config: Res<CampaignModeConfig>,
    bot_state: Res<BotConnectingState>,
    mut back_event: EventWriter<BackEvent>,
) {
    if is_typing_egui(&mut contexts) {
        // not a global interaction
        return;
    }

    if *bot_state == BotConnectingState::None {
        // select
        if settings
            .keybindings
            .menu_select
            .any(|code| input.just_pressed(code))
        {
            let allow_bot_connections = match config.selected_gamemode {
                // allow connectign bot in creative and challenge mode
                GameMode::Challenge | GameMode::Creative | GameMode::Zen => true,
                // only allow bot connections in supported campaign levels
                GameMode::Campaign => campaign_config.level.get_bot_type().is_some(),
            };
            click_event.send(UiClickEvent);
            start_game_internal(
                &mut settings,
                settings_bank_a,
                &config,
                &mut click_event,
                commands,
                allow_bot_connections,
                &campaign_config,
            );
        }
        // back
        if settings
            .keybindings
            .menu_back
            .any(|code| input.just_pressed(code))
        {
            click_event.send(UiClickEvent);
            back_event.send(BackEvent);
        }
    }
}

fn handle_back(config: Res<GameConfig>, mut setup_state: ResMut<NextState<SetupMenuState>>) {
    setup_state.set(match config.selected_gamemode {
        GameMode::Campaign => {
            // the campaign mode has to go back to level selection
            SetupMenuState::CampaignLevelSelect
        }
        _ => SetupMenuState::GameModeSelect,
    });
}
