use super::{theme::*, UiClickEvent};
use crate::{
    i18n::Localization,
    prelude::*,
    settings::{ActiveSettingsBank, PersistentSettingsBank},
    AppMenuState,
};
use bevy::prelude::*;
use bevy_egui::{
    egui::{self, FontId, RichText},
    EguiContexts,
};

pub mod audio;
pub mod controls;
pub mod general;
pub mod graphics;

use controls::SetKeybindLogic;

pub struct SettingsMenuPlugin;
impl Plugin for SettingsMenuPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (setting_menu_system, controls::keyboard_input)
                .run_if(in_state(AppMenuState::SettingsMenu)),
        );
        app.add_systems(
            OnExit(AppMenuState::SettingsMenu),
            controls::reset_keyboard_input,
        );

        app.add_systems(
            Update,
            (controls::key_remap_popup).run_if(in_state(KeyRemap::Listening)),
        );
        app.init_state::<KeyRemap>();
        app.init_resource::<SettingsTabs>();
        app.init_resource::<SetKeybindLogic>();

        app.init_resource::<RestartRequiredPopup>();
        app.add_systems(Update, show_restart_popup);
    }
}

#[derive(States, Default, Debug, Hash, PartialEq, Eq, Clone)]
pub enum KeyRemap {
    /// The User is currently not remapping a keybinding
    #[default]
    Deaf,
    /// The User is currently remapping a keybinding
    Listening,
}

#[derive(Default, Eq, PartialEq, Resource)]
pub enum SettingsTabs {
    #[default]
    General,
    Audio,
    Graphics,
    Controls,
}

#[derive(Default, Eq, PartialEq, Resource)]
pub struct RestartRequiredPopup {
    shown: bool,
    // translation ID
    reason: String,
}

#[allow(clippy::too_many_arguments)]
pub fn setting_menu_system(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut app_menu_state: ResMut<NextState<AppMenuState>>,
    remap_state: ResMut<NextState<KeyRemap>>,
    mut selected_tab: ResMut<SettingsTabs>,
    game_settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut set_key_bind_logic: ResMut<SetKeybindLogic>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
    restart_required: ResMut<RestartRequiredPopup>,
) {
    let localization = localization.into_inner();

    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    egui::TopBottomPanel::top("settings-top-bar")
        .frame(my_frame)
        .show_separator_line(false)
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;
            egui::ScrollArea::horizontal()
                .scroll_bar_visibility(egui::scroll_area::ScrollBarVisibility::AlwaysHidden)
                .show(ui, |ui| {
                    ui.horizontal(|ui| {
                        let mut clicked = false;
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::General,
                                RichText::new(localization.translate("general")).font(FontId::new(
                                    window.y * HEADING_SIZE,
                                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                )),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Audio,
                                RichText::new(localization.translate("audio")).font(FontId::new(
                                    window.y * HEADING_SIZE,
                                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                )),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Graphics,
                                RichText::new(localization.translate("graphics")).font(
                                    FontId::new(
                                        window.y * HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ),
                                ),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Controls,
                                RichText::new(localization.translate("controls")).font(
                                    FontId::new(
                                        window.y * HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ),
                                ),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if clicked {
                            click_event.send(UiClickEvent);
                        }

                        //ui.add(egui::Separator::default().horizontal().spacing(69.));
                        // ui.spacing();
                    });
                });

            ui.separator();
        });

    // bottom panel
    egui::TopBottomPanel::bottom("settings-bottom-bar")
        .show_separator_line(false)
        .frame(my_frame)
        .show(egui, |ui| {
            ui.style_mut().text_styles.insert(
                egui::TextStyle::Button,
                egui::FontId::new(
                    window.y * SECONDARY_BUTTON_TEXT_SIZE,
                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                ),
            );
            if ui
                .add_sized(
                    (window * SECONDARY_BUTTON_SIZE).cast(),
                    egui::Button::new(
                        RichText::new(localization.translate("save-and-exit"))
                            .color(BUTTON_TEXT_COLOR),
                    )
                    .fill(BUTTON_COLOR),
                )
                .clicked()
            {
                // apply settings from bank b to a
                // this saves all of them to disk
                **settings_bank_a = game_settings.clone();

                // return to main menu
                app_menu_state.set(AppMenuState::MainMenu);
                click_event.send(UiClickEvent);
            }
        });

    // center panel
    egui::CentralPanel::default()
        .frame(my_frame)
        // .show_separator_line(false) // doesn't work
        .show(egui, |ui| match selected_tab.as_mut() {
            SettingsTabs::General => {
                set_key_bind_logic.show_dialog = false;
                general::general_tab(
                    ui,
                    window,
                    game_settings,
                    localization,
                    click_event,
                    restart_required,
                )
            }
            SettingsTabs::Audio => {
                set_key_bind_logic.show_dialog = false;
                audio::audio_tab(ui, window, game_settings, localization)
            }
            SettingsTabs::Graphics => {
                set_key_bind_logic.show_dialog = false;
                graphics::graphics_tab(ui, window, game_settings, localization, click_event)
            }
            SettingsTabs::Controls => controls::keybindings_tab(
                ui,
                window,
                game_settings,
                set_key_bind_logic,
                localization,
                click_event,
                remap_state,
            ),
        });
}

/// Displays a modal popup window
/// The window can be dismissed by clicking the x button or the backdrop
/// the id is used for the backdrop area
pub fn modal_popup<R>(
    ctx: &egui::Context,
    id: &str,
    title: &str,
    show_dialog: &mut bool,
    add_contents: impl FnOnce(&mut egui::Ui) -> R,
) -> Option<egui::InnerResponse<Option<R>>> {
    let my_frame = egui::Frame {
        fill: POPUP_COLOR_BG,
        stroke: egui::Stroke::new(1.5, POPUP_STROKE_COLOR),
        corner_radius: POPUP_RADIUS,
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        ..Default::default()
    };

    let screen_size = ctx.input(|i| i.screen_rect);
    if *show_dialog {
        egui::Area::new(id.to_string().into())
            .order(egui::Order::Background)
            .show(ctx, |ui| {
                egui::Frame {
                    fill: POPUP_COLOR_BG.linear_multiply(0.7),
                    ..Default::default()
                }
                .show(ui, |ui| {
                    if ui
                        .allocate_rect(screen_size, egui::Sense::click())
                        .clicked()
                    {
                        *show_dialog = false;
                    }
                });
            });
    }

    egui::Window::new(title)
        .frame(my_frame)
        .collapsible(false)
        .resizable(false)
        .open(show_dialog)
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .order(egui::Order::Foreground)
        .show(ctx, add_contents)
}

pub fn show_restart_popup(
    mut contexts: EguiContexts,
    mut state: ResMut<RestartRequiredPopup>,
    localization: Res<Localization>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let reason = state.reason.to_string();

    modal_popup(
        egui,
        "restart-popup",
        &localization.translate("restart-required"),
        &mut state.shown,
        |ui| {
            ui.style_mut().interaction.selectable_labels = false;
            ui.label(localization.translate(&reason));
        },
    );
}
