use bevy::prelude::*;
use bevy_egui::egui::Pos2;
use bevy_egui::{egui, EguiContexts};
use std::collections::VecDeque;
use std::time::Duration;

use crate::game::bot::{
    BotActionOverflowEvent, BotDataErrorEvent, BotTimeoutEvent, ConnectionLostEvent,
};
use crate::i18n::{Localization, Translate};

use super::theme::{
    INNER_MARGIN, INV_SLOT_BG, INV_SLOT_RADIUS, INV_SLOT_STROKE, INV_SLOT_STROKE_HOVER,
    OUTER_MARGIN,
};

/// Time to retain a notification
const NOTIFICATIONS_TIME: Duration = Duration::from_millis(5000);
/// Length of fade-out animation
const NOTIFICATIONS_FADE_OUT_TIME: Duration = Duration::from_millis(750);
/// Min width of notification
const NOTIFICATIONS_MIN_WIDTH: f32 = 240.0;
/// Ratio of screen to use for width.
const NOTIFICATIONS_WIDTH_RATIO: f32 = 0.4;
/// Limit notification count to this number
const NOTIFICATION_MAX_COUNT: usize = 5;

pub struct NotificationsPlugin;
impl Plugin for NotificationsPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (handle_other_notification_events, notification_handler).chain(),
        );
    }
}

fn handle_other_notification_events(
    mut notifications: EventWriter<NotificationEvent>,
    mut bot_conn_lost: EventReader<ConnectionLostEvent>,
    mut bot_bad_data: EventReader<BotDataErrorEvent>,
    mut bot_overflow_data: EventReader<BotActionOverflowEvent>,
    mut bot_timeout_event: EventReader<BotTimeoutEvent>,
) {
    if !bot_conn_lost.is_empty() {
        bot_conn_lost.clear();
        notifications.send(NotificationEvent::Translation("notification-bot-conn_lost"));
    }
    if !bot_bad_data.is_empty() {
        bot_bad_data.clear();
        notifications.send(NotificationEvent::Translation("notification-bot-bad_data"));
    }
    if !bot_overflow_data.is_empty() {
        bot_overflow_data.clear();
        notifications.send(NotificationEvent::Translation(
            "notification-bot-action_overflow",
        ));
    }
    if !bot_timeout_event.is_empty() {
        bot_timeout_event.clear();
        notifications.send(NotificationEvent::Translation(
            "notification-bot-action_timeout",
        ));
    }
}

/// Notification Event. Send to display events. Might not be registered.
#[derive(Event, Debug, Clone)]
pub enum NotificationEvent {
    Text(String),
    Translation(&'static str),
}

struct RenderedNotification {
    content: NotificationEvent,
    received: Duration,
}

fn notification_handler(
    mut contexts: EguiContexts,
    localization: Res<Localization>,
    windows: Query<&Window>,
    mut notification_event: EventReader<NotificationEvent>,
    time: Res<Time<Real>>,
    mut notifications: Local<VecDeque<RenderedNotification>>,
) {
    for msg in notification_event.read() {
        notifications.push_front(RenderedNotification {
            content: msg.clone(),
            received: time.elapsed(),
        });
    }
    while let Some(oldest) = notifications.back() {
        // remove last notification, when limit exceeded or the oldest notification is old.
        if notifications.len() <= NOTIFICATION_MAX_COUNT
            && time.elapsed() - oldest.received < NOTIFICATIONS_TIME
        {
            break;
        }
        notifications.pop_back();
    }

    let Ok(window) = windows.get_single() else {
        return;
    };

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.width() == 0.0 || window.height() == 0.0 {
        return;
    }
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let width = egui.screen_rect().width() - OUTER_MARGIN * 2.0;

    egui::Area::new("notifications-area".into())
        .order(egui::Order::Foreground)
        .pivot(egui::Align2::CENTER_TOP)
        .fixed_pos(Pos2::new(window.width() / 2.0, OUTER_MARGIN))
        .show(egui, |ui| {
            notifications.retain(|msg| -> bool {
                let response = ui.add(NotificationCard {
                    opacity: ((NOTIFICATIONS_TIME - (time.elapsed() - msg.received)).as_secs_f32()
                        / NOTIFICATIONS_FADE_OUT_TIME.as_secs_f32())
                    .clamp(0.0, 1.0),
                    width: width
                        .min(NOTIFICATIONS_MIN_WIDTH.max(width * NOTIFICATIONS_WIDTH_RATIO)),
                    text: match &msg.content {
                        NotificationEvent::Text(text) => text.to_string(),
                        NotificationEvent::Translation(id) => localization.translate(id),
                    },
                });

                ui.add_space(OUTER_MARGIN);
                !response.clicked()
            });
        });
}

struct NotificationCard {
    text: String,
    width: f32,
    opacity: f32,
}

impl egui::Widget for NotificationCard {
    fn ui(self, ui: &mut egui::Ui) -> egui::Response {
        // set opacity
        ui.set_opacity(self.opacity);

        // manually create the frame, to make it clickable
        let mut prep = egui::Frame::group(ui.style())
            .corner_radius(INV_SLOT_RADIUS)
            .fill(INV_SLOT_BG)
            .inner_margin(egui::Margin::same(INNER_MARGIN as i8))
            .stroke(egui::Stroke::new(2.0, INV_SLOT_STROKE))
            .begin(ui);
        // manually add the content
        {
            let ui = &mut prep.content_ui;
            ui.style_mut().interaction.selectable_labels = false;
            ui.add_sized([self.width, 20.0], egui::Label::new(self.text));
        };
        let content_with_margin =
            prep.content_ui.min_rect() + (prep.frame.inner_margin + prep.frame.outer_margin);
        let alloc = ui.allocate_rect(content_with_margin, egui::Sense::click());
        if alloc.hovered() {
            prep.frame.fill = INV_SLOT_STROKE_HOVER;
        }
        prep.paint(ui);
        alloc
    }
}
