use std::collections::BTreeMap;

use about::AboutAppPlugin;
use app_loading::LoadingScreenPlugin;
use bevy::{core_pipeline::tonemapping::Tonemapping, prelude::*, window::WindowResized};
use bevy_egui::{
    egui::{FontFamily, FontId},
    EguiContexts, EguiPlugin,
};
use commons::CommonsPlugin;
use errors::show_error_screen;
use font_loader::set_fallback_fonts;
use help::HelpAppPlugin;
use in_game::InGameUiPlugin;
use main_menu::MainMenuPlugin;
use menu_bg::MenuBackgroundPlugin;
use notifications::NotificationsPlugin;
use settings_menu::SettingsMenuPlugin;
use setup_menu::SetupMenuPlugin;
use theme::{MONOSPACE_FONT_NAME, PARAGRAPH_FONT_NAME, TITLE_FONT_NAME};
use utils::{update_scroll_position, Scrollable};

use crate::{game::asset_loading::LACManager, settings::ActiveSettingsBank, AppState};

use self::font_loader::AppFonts;

pub mod about;
mod app_loading;
pub mod commons;
pub mod errors;
pub mod font_loader;
pub mod help;
pub mod in_game;
pub mod main_menu;
#[cfg(feature = "material-debugger")]
pub mod material_debugger;
mod menu_bg;
pub mod notifications;
pub mod settings_menu;
pub mod setup_menu;
pub mod theme;
pub mod utils;

pub const WEBSITE_BASE: &str = "https://terratactician-expandoria.codeberg.page";

/// Can not be used as a run_if system, because some traits are not satisfied.
pub fn is_using_egui(contexts: &mut EguiContexts) -> bool {
    let Some(egui) = contexts.try_ctx_mut() else {
        return true;
    };
    egui.is_using_pointer()
        || egui.is_pointer_over_area()
        || egui.is_context_menu_open()
        || egui.wants_keyboard_input()
}

pub fn is_typing_egui(contexts: &mut EguiContexts) -> bool {
    let Some(egui) = contexts.try_ctx_mut() else {
        return true;
    };
    egui.is_using_pointer() || egui.is_context_menu_open() || egui.wants_keyboard_input()
}

#[derive(Default)]
pub struct UiPlugin;

#[derive(Event)]
pub struct BackEvent;

impl Plugin for UiPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<AppFonts>();
        app.add_systems(Startup, set_fallback_fonts);

        app.add_event::<UiClickEvent>();
        app.add_event::<BackEvent>();

        app.init_resource::<WasUsingEgui>();

        app.add_plugins(EguiPlugin)
            .add_plugins(MainMenuPlugin)
            .add_plugins(AboutAppPlugin)
            .add_plugins(HelpAppPlugin)
            .add_plugins(SettingsMenuPlugin)
            .add_plugins(SetupMenuPlugin)
            .add_plugins(InGameUiPlugin)
            .add_plugins(CommonsPlugin)
            .add_plugins(LoadingScreenPlugin)
            .add_plugins(MenuBackgroundPlugin)
            .add_plugins(NotificationsPlugin);

        app.add_systems(Update, show_error_screen.run_if(in_state(AppState::Error)));
        app.add_systems(Update, reset_areas_on_resize);

        app.add_systems(
            Update,
            update_cam_settings.run_if(resource_changed::<ActiveSettingsBank>),
        );

        app.add_systems(PostUpdate, update_using_egui);
        app.add_systems(
            PreUpdate,
            apply_font_style.run_if(not(in_state(AppState::LoadApp))),
        );

        app.add_systems(
            Update,
            update_scroll_position.run_if(any_with_component::<Scrollable>),
        );

        #[cfg(feature = "material-debugger")]
        {
            use material_debugger::debug_material;
            app.add_systems(Update, debug_material);
        }

        #[cfg(feature = "ui-debugger")]
        {
            app.add_plugins(bevy::dev_tools::ui_debug_overlay::DebugUiPlugin)
                .add_systems(Update, toggle_overlay);
        }
    }
}

#[derive(Debug, Resource, Default)]
pub struct WasUsingEgui {
    pub using: bool,
    pub typing: bool,
}

fn update_using_egui(mut contexts: EguiContexts, mut was_using: ResMut<WasUsingEgui>) {
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    was_using.using = egui.is_using_pointer()
        || egui.is_pointer_over_area()
        || egui.is_context_menu_open()
        || egui.wants_keyboard_input();
    was_using.typing =
        egui.is_using_pointer() || egui.is_context_menu_open() || egui.wants_keyboard_input();
}

fn apply_font_style(mut contexts: EguiContexts) {
    use bevy_egui::egui::style::TextStyle::*;
    let text_styles: BTreeMap<_, _> = [
        (
            Heading,
            FontId::new(30.0, FontFamily::Name(TITLE_FONT_NAME.into())),
        ),
        (
            Body,
            FontId::new(18.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Button,
            FontId::new(14.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Small,
            FontId::new(12.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Monospace,
            FontId::new(14.0, FontFamily::Name(MONOSPACE_FONT_NAME.into())),
        ),
    ]
    .into();
    contexts
        .ctx_mut()
        .style_mut(move |style| style.text_styles = text_styles.clone());
}

#[derive(Clone)]
struct WindowSizeData {
    height: f32,
    width: f32,
}

/// number of percent the window size has to change
/// for the egui areas to be reset
/// the lower this value is, the lower the resize changes have to be
const RESIZE_RESET_DISTANCE: f32 = 0.2;

/// resets the area state when the window is resized a given distance
/// prevents weird visual glitches
fn reset_areas_on_resize(
    mut contexts: EguiContexts,
    mut events: EventReader<WindowResized>,
    mut size: Local<Option<WindowSizeData>>,
) {
    if events.is_empty() {
        return;
    }
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    for e in events.read() {
        let old_size = (*size).clone();
        *size = Some(WindowSizeData {
            height: e.height,
            width: e.width,
        });
        if let Some(old_size) = old_size {
            let dx = (old_size.width - e.width).abs() / old_size.width;
            let dy = (old_size.height - e.height).abs() / old_size.height;

            if dx <= RESIZE_RESET_DISTANCE && dy <= RESIZE_RESET_DISTANCE {
                // skip reset
                continue;
            }
        }
        // reset if the delta is large enough, or this is the first resize
        egui.memory_mut(|mem| mem.reset_areas());
    }
}

fn update_cam_settings(
    settings: Res<ActiveSettingsBank>,
    mut cams: Query<(&mut Camera, &mut Tonemapping), With<Camera2d>>,
) {
    for (mut cam, mut tonemapping) in cams.iter_mut() {
        cam.hdr = settings.graphics.hdr;
        *tonemapping = if settings.graphics.hdr {
            Tonemapping::default()
        } else {
            Tonemapping::None
        };
    }
}

#[derive(Event)]
pub struct UiClickEvent;

#[allow(dead_code)]
#[derive(Debug, Component)]
/// Adds a equi Material Debugger to an Entity. This will do nothing if feature `material-debugger` is disabled.
pub struct DebugMaterial(pub &'static str);

#[cfg(feature = "ui-debugger")]
// The system that will enable/disable the debug outlines around the nodes
fn toggle_overlay(
    input: Res<ButtonInput<KeyCode>>,
    mut options: ResMut<bevy::dev_tools::ui_debug_overlay::UiDebugOptions>,
) {
    info_once!("The debug outlines are enabled, press Space to turn them on/off");
    if input.just_pressed(KeyCode::Space) {
        // The toggle method will enable the debug_overlay if disabled and disable if enabled
        options.toggle();
    }
}
