use std::f32::consts::PI;

use bevy::{core_pipeline::tonemapping::Tonemapping, prelude::*};

use crate::{
    coordinates::CubeCoordinate,
    game::{
        controls::{cam::rotate_cam, PanOrbitCam},
        tiles::{assets::TileModels, TileType},
        visuals::{
            animations::{link_animations, start_tile_animations, update_animations_setteings},
            light::{despawn_light, setup_light, update_light_settings},
        },
    },
    prelude::*,
    settings::ActiveSettingsBank,
    AppState,
};

use super::theme::{BACKGROUND_ALPHA, BACKGROUND_ALPHA_HDR, BACKGROUND_COLOR};

const BACKGROUD_BLUR: f32 = 30.0;

const MENU_BG_DEFAULT_CAM_Y: f32 = PI / 2.0 * 0.75;

pub struct MenuBackgroundPlugin;

impl Plugin for MenuBackgroundPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_systems(
            OnEnter(AppState::AppMenu),
            (spawn_camera, spawn_scene, rotate_cam),
        );
        app.add_systems(Update, rotate_cam.run_if(in_state(AppState::AppMenu)));

        // configure lights
        app.add_systems(OnEnter(AppState::AppMenu), setup_light);
        app.add_systems(OnExit(AppState::AppMenu), despawn_light);
        app.add_systems(
            Update,
            update_light_settings
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::AppMenu)),
        );

        // configure animations
        app.add_systems(
            Update,
            (link_animations, start_tile_animations).run_if(in_state(AppState::AppMenu)),
        );
        app.add_systems(
            Update,
            update_animations_setteings
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::AppMenu)),
        );

        app.add_systems(OnEnter(AppState::AppMenu), spawn_overlay);
        app.add_systems(
            Update,
            update_overlay.run_if(resource_changed::<ActiveSettingsBank>),
        );
    }
}

// Render menu background overlay
fn spawn_overlay(mut commands: Commands, settings: Res<ActiveSettingsBank>) {
    #[cfg(target_os = "android")]
    let msaa = Msaa::Off;
    #[cfg(not(target_os = "android"))]
    let msaa = Msaa::default();
    commands.spawn((
        Camera2d,
        msaa,
        Transform::default(),
        if settings.graphics.hdr {
            Tonemapping::default()
        } else {
            Tonemapping::None
        },
        Camera {
            clear_color: ClearColorConfig::None,
            hdr: settings.graphics.hdr,
            order: 1,
            ..default()
        },
        IsDefaultUiCamera,
        StateScoped(AppState::AppMenu),
    ));
    commands.spawn((
        ZIndex(-1),
        Node {
            width: Val::Percent(100.),
            height: Val::Percent(100.),
            ..Default::default()
        },
        StateScoped(AppState::AppMenu),
        MenuBackgroundMarker,
        BackgroundColor(
            BACKGROUND_COLOR
                .cast()
                .with_alpha(if settings.graphics.hdr {
                    BACKGROUND_ALPHA_HDR
                } else {
                    BACKGROUND_ALPHA
                }),
        ),
    ));
}

#[derive(Component)]
struct MenuBackgroundMarker;
fn update_overlay(
    mut overlays: Query<&mut BackgroundColor, With<MenuBackgroundMarker>>,
    settings: Res<ActiveSettingsBank>,
) {
    for mut color in overlays.iter_mut() {
        color.0 = BACKGROUND_COLOR
            .cast()
            .with_alpha(if settings.graphics.hdr {
                BACKGROUND_ALPHA_HDR
            } else {
                BACKGROUND_ALPHA
            });
    }
}

fn spawn_camera(mut commands: Commands, settings: Res<ActiveSettingsBank>) {
    #[cfg(target_os = "android")]
    let msaa = Msaa::Off;
    #[cfg(not(target_os = "android"))]
    let msaa = Msaa::default();

    let mut panorbit = PanOrbitCam::default();
    panorbit.orbit.y = MENU_BG_DEFAULT_CAM_Y;
    panorbit.radius = 6.;
    use crate::game::visuals::blur::BlurSettings;
    commands.spawn((
        Camera3d::default(),
        msaa,
        Transform::default(),
        if settings.graphics.hdr {
            Tonemapping::default()
        } else {
            Tonemapping::None
        },
        Camera {
            hdr: settings.graphics.hdr,
            ..default()
        },
        panorbit,
        BlurSettings::new(BACKGROUD_BLUR),
        StateScoped(AppState::AppMenu),
    ));
}

fn spawn_scene(mut commands: Commands, models: Res<TileModels>, assets: Res<Assets<Gltf>>) {
    for (tile_type, pos) in [
        (TileType::Windmill, CubeCoordinate::new(0, 0, 0)),
        (TileType::Wheat, CubeCoordinate::new(0, 1, -1)),
        (TileType::Wheat, CubeCoordinate::new(-1, 1, 0)),
        (TileType::Grass, CubeCoordinate::new(1, 0, -1)),
        (TileType::SmallHouse, CubeCoordinate::new(1, -1, 0)),
        (TileType::Forest, CubeCoordinate::new(0, -1, 1)),
        (TileType::Forest, CubeCoordinate::new(-1, 0, 1)),
    ] {
        let scene = {
            let model_handle = models.0.get(&tile_type).expect(
                "The Loading Screen should have taken care of loading the model at this point.",
            );
            let gltf = assets
                .get(model_handle)
                .expect("The LoadingScreen should have ensured that the asset was loaded.");

            gltf.scenes
                .get(tile_type.get_asset_scene())
                .expect("The Scene should be available for the model")
                .clone_weak()
        };

        commands.spawn((
            SceneRoot(scene),
            Transform::from_translation(pos.into()),
            tile_type,
            StateScoped(AppState::AppMenu),
        ));
    }
}
