use std::collections::HashSet;

use bevy::prelude::*;
use bevy_egui::{egui, EguiContexts};

use super::DebugMaterial;

/// feature gated material debug window
pub fn debug_material(
    mut contexts: EguiContexts,
    todebug: Query<(&DebugMaterial, &MeshMaterial3d<StandardMaterial>), With<DebugMaterial>>,
    mut materials: ResMut<Assets<StandardMaterial>>,
) {
    use std::f32::consts::PI;
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // track debugged materials
    let mut rendered = HashSet::new();

    for (debug_name, material) in todebug.iter() {
        // skip already rendered material debuggers.
        if rendered.contains(&material.id()) {
            continue;
        }
        rendered.insert(material.id());

        egui::Window::new(format!("Material Debugger - {}", debug_name.0))
            .id(format!("material-debugger-{}", material.id()).into())
            .default_open(false)
            .show(egui, |ui| {
                if let Some(mat) = materials.get_mut(material) {
                    egui::Grid::new("my_grid")
                        .num_columns(2)
                        .spacing([40.0, 4.0])
                        .striped(true)
                        .show(ui, |ui| {
                            let mut base_color = mat.base_color.to_srgba().to_f32_array();
                            ui.label("Base color");
                            if ui
                                .color_edit_button_rgba_unmultiplied(&mut base_color)
                                .changed()
                            {
                                mat.base_color = Color::srgba(
                                    base_color[0],
                                    base_color[1],
                                    base_color[2],
                                    base_color[3],
                                );
                            }
                            ui.end_row();

                            ui.label("Emissive color");
                            let mut emissive = mat.emissive.to_f32_array();
                            if ui
                                .color_edit_button_rgba_unmultiplied(&mut emissive)
                                .changed()
                            {
                                mat.emissive = LinearRgba::new(
                                    emissive[0],
                                    emissive[1],
                                    emissive[2],
                                    emissive[3],
                                );
                            }

                            ui.end_row();

                            /* egui::ComboBox::from_label("Base Colorsive Channel")
                            .selected_text(format!("{:?}", mat.emissive_channel))
                            .show_ui(ui, |ui| {
                                ui.selectable_value(&mut mat.emissive_channel, UvChannel::Uv0, "Uv0");
                                ui.selectable_value(&mut mat.emissive_channel, UvChannel::Uv1, "Uv1");
                            }); */

                            ui.label("Emmissive exposure weight");
                            ui.add(egui::Slider::new(
                                &mut mat.emissive_exposure_weight,
                                0.0..=1.0,
                            ));
                            ui.end_row();

                            ui.label("Perceptual roughness");
                            ui.add(egui::Slider::new(&mut mat.perceptual_roughness, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Metallic");
                            ui.add(egui::Slider::new(&mut mat.metallic, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Reflectance");
                            ui.add(egui::Slider::new(&mut mat.reflectance, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Diffuse transmission");
                            ui.add(egui::Slider::new(&mut mat.diffuse_transmission, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Specular transmission");
                            ui.add(egui::Slider::new(&mut mat.specular_transmission, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Thickness");
                            ui.add(egui::Slider::new(&mut mat.thickness, 0.0..=10.0));
                            ui.end_row();

                            ui.label("Ior");
                            ui.add(egui::Slider::new(&mut mat.ior, 1.0..=3.0));
                            ui.end_row();

                            ui.label("Attenuation distance");
                            ui.add(
                                egui::Slider::new(
                                    &mut mat.attenuation_distance,
                                    0.0..=f32::INFINITY,
                                )
                                .logarithmic(true),
                            );
                            ui.end_row();

                            ui.label("Attenuation color");
                            let mut attenuation_color =
                                mat.attenuation_color.to_srgba().to_f32_array();
                            if ui
                                .color_edit_button_rgba_unmultiplied(&mut attenuation_color)
                                .changed()
                            {
                                mat.attenuation_color = Color::srgba(
                                    attenuation_color[0],
                                    attenuation_color[1],
                                    attenuation_color[2],
                                    attenuation_color[3],
                                );
                            }
                            ui.end_row();

                            ui.label("Flip normal map y");
                            ui.checkbox(&mut mat.flip_normal_map_y, "fliped");
                            ui.end_row();

                            ui.label("Clearcoat");
                            ui.add(egui::Slider::new(&mut mat.clearcoat, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Clearcoat perceptual roughness");
                            ui.add(egui::Slider::new(
                                &mut mat.clearcoat_perceptual_roughness,
                                0.0..=1.0,
                            ));
                            ui.end_row();

                            ui.label("Anisotropy strength");
                            ui.add(egui::Slider::new(&mut mat.anisotropy_strength, 0.0..=1.0));
                            ui.end_row();

                            ui.label("Anisotropy rotation");
                            ui.add(egui::Slider::new(
                                &mut mat.anisotropy_rotation,
                                0.0..=2.0 * PI,
                            ));
                            ui.end_row();

                            ui.label("Double sided");
                            ui.checkbox(&mut mat.double_sided, "double");
                            ui.end_row();

                            //cull_mode

                            ui.label("Unlit");
                            ui.checkbox(&mut mat.unlit, "unlit");
                            ui.end_row();

                            ui.label("Fog enabled");
                            ui.checkbox(&mut mat.fog_enabled, "enabled");
                            ui.end_row();

                            ui.label("AlphaMode");
                            egui::ComboBox::from_id_salt("AlphaMode")
                                .selected_text(format!("{:?}", mat.alpha_mode))
                                .show_ui(ui, |ui| {
                                    ui.selectable_value(&mut mat.alpha_mode, AlphaMode::Add, "Add");
                                    ui.selectable_value(
                                        &mut mat.alpha_mode,
                                        AlphaMode::AlphaToCoverage,
                                        "AlphaToCoverage",
                                    );
                                    ui.selectable_value(
                                        &mut mat.alpha_mode,
                                        AlphaMode::Blend,
                                        "Blend",
                                    );
                                    ui.selectable_value(
                                        &mut mat.alpha_mode,
                                        AlphaMode::Multiply,
                                        "Multiply",
                                    );
                                    ui.selectable_value(
                                        &mut mat.alpha_mode,
                                        AlphaMode::Opaque,
                                        "Opaque",
                                    );
                                    ui.selectable_value(
                                        &mut mat.alpha_mode,
                                        AlphaMode::Premultiplied,
                                        "Premultiplied",
                                    );
                                });
                            ui.end_row();

                            ui.label("Depth bias");
                            ui.add(egui::Slider::new(&mut mat.depth_bias, -10.0..=10.0));
                            ui.end_row();

                            ui.label("Parallax depth scale");
                            ui.add(egui::Slider::new(&mut mat.parallax_depth_scale, 0.0..=1.0));
                            ui.end_row();

                            //parallax_mapping_method

                            ui.label("Max parallax layer count");
                            ui.add(egui::Slider::new(
                                &mut mat.max_parallax_layer_count,
                                0.0..=32.0,
                            ));
                            ui.end_row();

                            ui.label("Lightmap exposure");
                            ui.add(
                                egui::Slider::new(&mut mat.lightmap_exposure, 0.0..=10.0).text(""),
                            );
                            ui.end_row();

                            ui.label("OpaqueRendererMethod");
                            egui::ComboBox::from_id_salt("OpaqueRendererMethod")
                                .selected_text(format!("{:?}", mat.opaque_render_method))
                                .show_ui(ui, |ui| {
                                    ui.selectable_value(
                                        &mut mat.opaque_render_method,
                                        bevy::pbr::OpaqueRendererMethod::Auto,
                                        "Auto",
                                    );
                                    ui.selectable_value(
                                        &mut mat.opaque_render_method,
                                        bevy::pbr::OpaqueRendererMethod::Deferred,
                                        "Deferred",
                                    );
                                    ui.selectable_value(
                                        &mut mat.opaque_render_method,
                                        bevy::pbr::OpaqueRendererMethod::Forward,
                                        "Forward",
                                    );
                                });
                            ui.end_row();
                        });
                }
            });
    }
}
