use bevy::prelude::*;
use bevy_egui::{
    egui::{self, FontId, RichText},
    EguiContexts,
};

use crate::{
    game::GamePauseState,
    i18n::Localization,
    settings::{ActiveSettingsBank, PersistentSettingsBank},
    ui::{
        settings_menu::{
            audio,
            controls::{self, SetKeybindLogic},
            general, graphics, KeyRemap, RestartRequiredPopup, SettingsTabs,
        },
        theme::{
            BACKGROUND_ALPHA, BACKGROUND_COLOR, BUTTON_COLOR, BUTTON_TEXT_COLOR, HEADING_SIZE,
            PARAGRAPH_FONT_NAME, SECONDARY_BUTTON_SIZE, SECONDARY_BUTTON_TEXT_SIZE,
            TITLE_FONT_NAME,
        },
        UiClickEvent,
    },
    AppState,
};

use super::{pause::PauseUiState, Cast, Translate};

pub struct InGameSettingsPlugin;
impl Plugin for InGameSettingsPlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<InGameSettingsState>();
        app.add_systems(
            OnExit(InGameSettingsState::Shown),
            controls::reset_keyboard_input,
        );
        app.add_systems(
            Update,
            (setting_menu_system, controls::keyboard_input)
                .run_if(in_state(AppState::InGame))
                .run_if(in_state(InGameSettingsState::Shown)),
        );
    }
}

#[derive(SubStates, Default, Debug, Clone, Hash, Eq, PartialEq)]
#[source(GamePauseState = GamePauseState::Paused)]
pub enum InGameSettingsState {
    /// the settings screen is visible and the game is paused
    Shown,
    /// the game is paused, but settings screen is not visible
    #[default]
    Hidden,
}

#[allow(clippy::too_many_arguments)]
pub fn setting_menu_system(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut pauseui: ResMut<NextState<PauseUiState>>,
    mut settingsstate: ResMut<NextState<InGameSettingsState>>,
    mut selected_tab: ResMut<SettingsTabs>,
    game_settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut set_key_bind_logic: ResMut<SetKeybindLogic>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
    remap_state: ResMut<NextState<KeyRemap>>,
    restart_required: ResMut<RestartRequiredPopup>,
) {
    let localization = localization.into_inner();

    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    egui::TopBottomPanel::top("settings-top-bar")
        .frame(my_frame)
        .show_separator_line(false)
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;
            egui::ScrollArea::horizontal()
                .scroll_bar_visibility(egui::scroll_area::ScrollBarVisibility::AlwaysHidden)
                .show(ui, |ui| {
                    ui.horizontal(|ui| {
                        let mut clicked = false;
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::General,
                                RichText::new(localization.translate("general")).font(FontId::new(
                                    window.y * HEADING_SIZE,
                                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                )),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Audio,
                                RichText::new(localization.translate("audio")).font(FontId::new(
                                    window.y * HEADING_SIZE,
                                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                )),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Graphics,
                                RichText::new(localization.translate("graphics")).font(
                                    FontId::new(
                                        window.y * HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ),
                                ),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if ui
                            .selectable_value(
                                selected_tab.as_mut(),
                                SettingsTabs::Controls,
                                RichText::new(localization.translate("controls")).font(
                                    FontId::new(
                                        window.y * HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ),
                                ),
                            )
                            .clicked()
                        {
                            clicked = true;
                        };
                        if clicked {
                            click_event.send(UiClickEvent);
                        }

                        //ui.add(egui::Separator::default().horizontal().spacing(69.));
                        // ui.spacing();
                    });
                });

            ui.separator();
        });

    // bottom panel
    egui::TopBottomPanel::bottom("settings-bottom-bar")
        .show_separator_line(false)
        .frame(my_frame)
        .show(egui, |ui| {
            ui.style_mut().text_styles.insert(
                egui::TextStyle::Button,
                egui::FontId::new(
                    window.y * SECONDARY_BUTTON_TEXT_SIZE,
                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                ),
            );
            if ui
                .add_sized(
                    (window * SECONDARY_BUTTON_SIZE).cast(),
                    egui::Button::new(
                        RichText::new(localization.translate("save-and-back"))
                            .color(BUTTON_TEXT_COLOR),
                    )
                    .fill(BUTTON_COLOR),
                )
                .clicked()
            {
                // apply settings from bank b to a
                // this saves all of them to disk
                **settings_bank_a = game_settings.clone();

                // return to main menu

                pauseui.set(PauseUiState::Shown);
                settingsstate.set(InGameSettingsState::Hidden);
                click_event.send(UiClickEvent);
            }
        });

    // center panel
    egui::CentralPanel::default()
        .frame(my_frame)
        // .show_separator_line(false) // doesn't work
        .show(egui, |ui| match selected_tab.as_mut() {
            SettingsTabs::General => {
                set_key_bind_logic.show_dialog = false;
                general::general_tab(
                    ui,
                    window,
                    game_settings,
                    localization,
                    click_event,
                    restart_required,
                )
            }
            SettingsTabs::Audio => {
                set_key_bind_logic.show_dialog = false;
                audio::audio_tab(ui, window, game_settings, localization)
            }
            SettingsTabs::Graphics => {
                set_key_bind_logic.show_dialog = false;
                graphics::graphics_tab(ui, window, game_settings, localization, click_event)
            }
            SettingsTabs::Controls => controls::keybindings_tab(
                ui,
                window,
                game_settings,
                set_key_bind_logic,
                localization,
                click_event,
                remap_state,
            ),
        });
}
