use bevy::{ecs::system::SystemParam, prelude::*, state::state::FreelyMutableState};
use bevy_egui::{
    egui::{self, Align2, FontFamily, FontId, RichText},
    EguiContexts,
};

use crate::{
    game::{
        controls::KeyboardEvent,
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
                CampaignGroups,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
            GameMode, GameModeSubStates,
        },
        metrics::{Metrics, MetricsRate, TargetMetrics},
        round::{RoundCounter, RoundSystemSupport, RoundTimer},
        time::{Game, GameTime},
        GamePauseState, GameSimSet,
    },
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        in_game::progress_bars::{
            spawn_progressbars_container, update_progressbars_colors, update_progressbars_progress,
            update_progressbars_text,
        },
        is_using_egui,
        theme::{
            BACKGROUND_ALPHA, BUTTON_COLOR, BUTTON_TEXT_COLOR, METRICS_BAR_FG_COLOR,
            METRICS_BAR_FG_COLOR_ATTENTION, PARAGRAPH_FONT_NAME, PRIMARY_BUTTON_SIZE,
            PRIMARY_BUTTON_TEXT_SIZE, SPACE_SIZE, TITLE_FONT_NAME,
        },
        UiClickEvent,
    },
    AppState,
};

use crate::ui::theme::{
    BETWEENROUNDS_BACKGROUND_COLOR, GAMEOVER_BACKGROUND_COLOR, GRID_TEXT_COLOR,
    OVERLAY_TITLE_TEXT_COLOR,
};

use super::{
    game_export::show_export_window,
    progress_bars::{ProgressBar, ProgressBarColors, ProgressBarContainer, ProgressBarText},
};

macro_rules! define_rounds_ui_plugin {
    ($({
        // gamemode in which to use bars
        gamemode: $gamemode:expr,
        // gamemode substates
        states: $states:ident,
        // state, in which the `BetweenRounds` screen is shown
        // has to be a substate of states
        between: $between:expr,
        // state, in which the `GameOver` screen is shown
        // has to be a substate of states
        $(end: $end:expr,)?
        // state in which the round progressbar is shown
        show_bar_in: $show_bar_in:expr,
        // system used to update the round progress bar
        update_round_bar: $update_round_bar:expr,
    }),*) => {
        pub struct RoundsUiPlugin;
        impl Plugin for RoundsUiPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                $(
                    app.add_systems(
                        Update,
                        (
                            show_round_transition_screen::<$states>.before(show_export_window),
                            skip_round_transition::<$states>,
                            keybinding_next_round::<$states>,
                        )
                            .run_if(in_state($between))
                            .run_if(in_state(GamePauseState::Paused)),
                    );
                    $(
                        app.add_systems(
                            Update,
                            show_game_over_screen::<$states>
                                .run_if(in_state($end))
                                .run_if(in_state(GamePauseState::Paused))
                                .before(show_export_window),
                        );
                    )?
                    app.add_systems(
                        Update,
                        $update_round_bar
                            .before(update_progressbars_text)
                            .before(update_progressbars_colors)
                            .before(update_progressbars_progress)
                            .after(GameSimSet::Simulation)
                            .run_if(in_state(ShowGameUi::On))
                            // HACK: prevent the round counter from being updated
                            // right before the round is over
                            // fixes the bug where the round counter was incremented
                            // before entering the between rounds screen
                            .run_if(still_in_state(GamePauseState::Running))
                            .run_if(in_state($show_bar_in))
                    );
                app.add_systems(
                    OnEnter($gamemode),
                    spawn_round_bar.after(spawn_progressbars_container)
                );
                )*
            }
        }
    }
}
define_rounds_ui_plugin![
    {
        gamemode: GameMode::Challenge,
        states: ChallengeGameState,
        between: ChallengeGameState::BetweenRounds,
        end: ChallengeGameState::GameOver,
        show_bar_in: ChallengeGameState::InRound,
        update_round_bar: timelimited_round_bar,
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        states: CampaignChallengeState,
        between: CampaignChallengeState::BetweenRounds,
        end: CampaignChallengeState::GameOver,
        show_bar_in: CampaignChallengeState::InRound,
        update_round_bar: timelimited_round_bar,
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        states: CampaignWettbewerbState,
        between: CampaignWettbewerbState::BetweenRounds,
        end: CampaignWettbewerbState::GameOver,
        show_bar_in: CampaignWettbewerbState::InRound,
        update_round_bar: timelimited_round_bar,
    },
    {
        gamemode: GameMode::Zen,
        states: ZenGameState,
        between: ZenGameState::BetweenRounds,
        show_bar_in: ZenGameState::InRound,
        update_round_bar: metricsbased_round_bar,
    }
];

/// Marker struct for Round-Progress progress bar
#[derive(Component)]
struct RoundProgressBar;

pub fn spawn_round_bar(
    mut commands: Commands,
    containers: Query<Entity, With<ProgressBarContainer>>,
    localization: Res<Localization>,
) {
    // we wouldn't know where to spawn the bars,
    // if there was more than one container
    let Ok(container) = containers.get_single() else {
        return;
    };

    commands
        .get_entity(container)
        .expect("We obtained the entity from the query, so it should exist")
        .with_children(|builder| {
            builder.spawn((
                ProgressBar(1.0),
                ProgressBarText(format!("{} 1", localization.translate("game-round"))),
                RoundProgressBar,
                ProgressBarColors::default(),
            ));
        });
}

/// shows the round progress as going down
/// indicating that time is running out
/// the progress is based on the round time
/// it also recolors the bar to show that the player is almost out of X
fn timelimited_round_bar(
    mut bars: Query<
        (
            &mut ProgressBar,
            &mut ProgressBarText,
            &mut ProgressBarColors,
        ),
        With<RoundProgressBar>,
    >,
    round_counter: Res<RoundCounter>,
    localization: Res<Localization>,
    round_timer: Res<RoundTimer>,
    gtime: Res<Time<Game>>,
) {
    let sub_secs = gtime.sub_secs();
    let round_progress =
        (round_timer.elapsed().as_secs_f32() + sub_secs) / round_timer.duration().as_secs_f32();

    for (mut progress, mut text, mut colors) in bars.iter_mut() {
        **progress = 1.0 - round_progress;
        **text = format!(
            "{} {}",
            localization.translate("game-round"),
            **round_counter + 1
        );
        // color the progressbar red if the time is almost up
        colors.bar_color = if round_progress > 0.9 {
            METRICS_BAR_FG_COLOR_ATTENTION.cast()
        } else {
            METRICS_BAR_FG_COLOR.cast()
        };
    }
}

/// shows the round progress as going up
/// based on the metrics collected in relation to the metrics target
fn metricsbased_round_bar(
    mut bars: Query<(&mut ProgressBar, &mut ProgressBarText), With<RoundProgressBar>>,
    round_counter: Res<RoundCounter>,
    localization: Res<Localization>,
    metrics: Res<Metrics>,
    target_metrics: Res<TargetMetrics>,
    metrics_rate: Res<MetricsRate>,
    gtime: Res<Time<Game>>,
) {
    let sub_secs = gtime.sub_secs();

    let food =
        ((metrics.food + sub_secs * metrics_rate.food) / target_metrics.food).clamp(0.0, 1.0);
    let materials = ((metrics.materials + sub_secs * metrics_rate.materials)
        / target_metrics.materials)
        .clamp(0.0, 1.0);
    let money =
        ((metrics.money + sub_secs * metrics_rate.money) / target_metrics.money).clamp(0.0, 1.0);
    let round_progress = (food + materials + money) / 3.0;

    for (mut progress, mut text) in bars.iter_mut() {
        **progress = round_progress;
        **text = format!(
            "{} {}",
            localization.translate("game-round"),
            **round_counter + 1
        );
    }
}

/// Detects mouse and touch input outside of egui elements
/// and continues to the next round if an event was detected
fn skip_round_transition<STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport>(
    mut contexts: EguiContexts,
    mouse_input: Res<ButtonInput<MouseButton>>,
    touches: Res<Touches>,
    mut next_state: ResMut<NextState<STATES>>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    // don't process events manually if the user interacted with the UI
    if is_using_egui(&mut contexts) {
        return;
    }
    // continue to next round if the player clicks anywhere on screen
    // with the mouse or on a touch screen
    if mouse_input.just_released(MouseButton::Left) || touches.any_just_released() {
        next_state.set(STATES::get_inround_state());
        click_event.send(UiClickEvent);
    }
}

/// shows the between rounds screen
fn show_round_transition_screen<
    STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport,
>(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut next_state: ResMut<NextState<STATES>>,
    localization: Res<Localization>,
    round_counter: Res<RoundCounter>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // draw background overlay
    let frame = egui::Frame {
        fill: BETWEENROUNDS_BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(egui, |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let big_text_size = window.y / 10.0;
    let localization = localization.into_inner();

    egui::Area::new("round-transition-overlay".into())
        .anchor(Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    RichText::new(localization.translate("game-round-end"))
                        .font(FontId::new(
                            big_text_size,
                            FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(OVERLAY_TITLE_TEXT_COLOR),
                );
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("game-round-finished"),
                        round_counter.0 + 1 - 1 // +1 to get ui round count -1 to get the finished round
                    ))
                    .color(GRID_TEXT_COLOR),
                );

                ui.add_space(SPACE_SIZE * window.y * 3.0);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("game-round-next"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    next_state.set(STATES::get_inround_state());
                    click_event.send(UiClickEvent);
                }
            });
        });
}

#[derive(SystemParam)]
struct GameOverData<'w> {
    app_state: ResMut<'w, NextState<AppState>>,
    pause_state: ResMut<'w, NextState<GamePauseState>>,
    click_event: EventWriter<'w, UiClickEvent>,
}

/// shows the end of game screen
fn show_game_over_screen<STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport>(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut game_state: ResMut<NextState<STATES>>,
    localization: Res<Localization>,
    round_counter: Res<RoundCounter>,
    mut ctrl: GameOverData,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // draw background overlay
    let frame = egui::Frame {
        fill: GAMEOVER_BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(egui, |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let big_text_size = window.y / 10.0;
    let localization = localization.into_inner();

    egui::Area::new("round-game-over".into())
        .anchor(Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    RichText::new(localization.translate("game-over"))
                        .font(FontId::new(
                            big_text_size,
                            FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(OVERLAY_TITLE_TEXT_COLOR),
                );
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("game-over-round"),
                        round_counter.0 + 1 // +1 to get the ui round count. number was not increased on gameover
                    ))
                    .color(GRID_TEXT_COLOR),
                );

                ui.add_space(SPACE_SIZE * window.y * 3.0);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                if let Some(spectate_state) = STATES::get_spectate_state() {
                    if ui
                        .add_sized(
                            (window * PRIMARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                egui::RichText::new(localization.translate("spectate"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        game_state.set(spectate_state);
                        ctrl.pause_state.set(GamePauseState::Running);
                        ctrl.click_event.send(UiClickEvent);
                    }
                }

                ui.add_space(window.y * SPACE_SIZE);

                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("exit-to-menu"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    ctrl.app_state.set(AppState::AppMenu);
                    ctrl.click_event.send(UiClickEvent);
                }
            });
        });
}

fn keybinding_next_round<STATES: GameModeSubStates + RoundSystemSupport + FreelyMutableState>(
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    mut next_state: ResMut<NextState<STATES>>,
    mut kb_event: EventWriter<KeyboardEvent>,
) {
    if settings
        .0
        .keybindings
        .next_round
        .any(|code| input.just_pressed(code))
    {
        next_state.set(STATES::get_inround_state());
        kb_event.send(KeyboardEvent);
    }
}
