use std::f32::consts::PI;

use bevy_egui::egui::{
    epaint::{PathShape, PathStroke},
    Color32, Pos2, Vec2,
};

/// rounded rectangle that with circular fill pattern
/// fillst the rectangle counter clock wise
pub struct ProgressRect {
    /// size of width and height in px
    pub size: f32,
    /// corner radius in px
    pub rounding: f32,
    /// number of segments per rounded corner
    /// the higher the rounder it looks
    /// however lower values should perform better
    pub rounding_segments: usize,
    /// how far the rect is filled
    /// [0..1]
    pub progress: f32,
    /// fill color to use
    pub fill: Color32,
}

impl ProgressRect {
    /// paint the rectangle
    /// use the offset to rotate or offset the points
    pub fn paint<F>(self, painter: &bevy_egui::egui::Painter, offset: F)
    where
        F: Fn(Pos2) -> Pos2,
    {
        if self.progress == 0.0 {
            return;
        }
        // top left
        ProgressRectPiece {
            size: self.size / 2.0,
            rounding: self.rounding,
            progress: (self.progress * 4.).clamp(0.0, 1.0),
            fill: self.fill,
            rounding_segments: self.rounding_segments,
        }
        .paint(painter, |pos| offset(Pos2::new(pos.y, -pos.x)));
        if self.progress >= 0.25 {
            // bottom left
            ProgressRectPiece {
                size: self.size / 2.0,
                rounding: self.rounding,
                progress: (self.progress * 4. - 1.0).clamp(0.0, 1.0),
                fill: self.fill,
                rounding_segments: self.rounding_segments,
            }
            .paint(painter, |pos| offset(Pos2::new(-pos.x, -pos.y)));
            if self.progress >= 0.5 {
                // bottom right
                ProgressRectPiece {
                    size: self.size / 2.0,
                    rounding: self.rounding,
                    progress: (self.progress * 4. - 2.0).clamp(0.0, 1.0),
                    fill: self.fill,
                    rounding_segments: self.rounding_segments,
                }
                .paint(painter, |pos| offset(Pos2::new(-pos.y, pos.x)));
                if self.progress >= 0.75 {
                    // top right
                    ProgressRectPiece {
                        size: self.size / 2.0,
                        rounding: self.rounding,
                        progress: (self.progress * 4. - 3.0).clamp(0.0, 1.0),
                        fill: self.fill,
                        rounding_segments: self.rounding_segments,
                    }
                    .paint(painter, &offset);
                }
            }
        }
    }
}

/// internal use only
/// fillable rounded quadrant
struct ProgressRectPiece {
    /// size of width and height in px
    pub size: f32,
    /// corner radius in px
    pub rounding: f32,
    /// how far the segment is filled
    /// [0..1]
    pub progress: f32,
    /// fill color to use
    pub fill: Color32,
    /// number of segments per rounded corner
    /// the higher the rounder it looks
    /// however lower values should perform better
    pub rounding_segments: usize,
}
impl ProgressRectPiece {
    pub fn paint<F>(self, painter: &bevy_egui::egui::Painter, offset: F)
    where
        F: Fn(Pos2) -> Pos2,
    {
        if self.progress == 0.0 {
            return;
        }

        let mut points = vec![Pos2::new(0.0, 0.0), Pos2::new(self.size, 0.0)];

        let side_length = self.size - self.rounding;
        // the rounding is only 90degrees of a circle
        // meaning that the circumference is 1/4 of the actual circumference
        let rounding_circumference = 2.0 * PI * self.rounding / 4.0;

        // the ProgressRectPiece describes only one section of the entire rect
        // made up of the top section, the rounding and the side section
        // because the widget has to be a square, the side length is the same
        let circumference = side_length * 2.0 + rounding_circumference;

        let progress_dist = circumference * self.progress;

        // right edge
        {
            // calculate the fraction of how far we are into the right edge
            // if the progress exceeds the right edge, it is capped to 1.0
            // to make sure that rounding entry point is properly displayed
            let edge_progress = (progress_dist / side_length).min(1.0);
            // NOTE: move the point at least one pixel to the top to prevent visual glitches
            // when the polygon nearly turns into a line
            points.push(Pos2::new(
                self.size,
                -(edge_progress * side_length).max(1.0),
            ));
        }

        // rounding
        if progress_dist > side_length {
            // determine how far we are into the rounding
            // the value can be 1.0 at most, so it wont exceed the quarter circle
            let rounding_progress =
                ((progress_dist - side_length) / rounding_circumference).min(1.0);

            let rounding_center = Pos2::new(side_length, -side_length);

            let step_size = 1.0 / self.rounding_segments as f32;

            for step in 1..self.rounding_segments {
                let current_progress = step as f32 * step_size;
                if current_progress > rounding_progress {
                    // ahead of animation,
                    // no need to render this point
                    break;
                }
                let current_angle = PI / 2.0 * current_progress;
                let radius = Vec2::new(
                    self.rounding * current_angle.cos(),
                    -self.rounding * current_angle.sin(),
                );
                points.push(rounding_center + radius);
            }
        }

        // top edge
        if progress_dist >= side_length + rounding_circumference {
            // add the rounding eexit point
            points.push(Pos2::new(side_length, -self.size));

            // render the second point on the top edge
            // determine the position on the top edge,
            // the right side and rounding have to be subtracted from the circle,
            // to get the correct number
            // the value is then clamped to 1.0 so it wont exceed the object bounds
            let edge_progress =
                ((progress_dist - (side_length + rounding_circumference)) / side_length).min(1.0);
            // NOTE: the x value has to be inverted, as it moves from the right to the left
            points.push(Pos2::new(
                side_length - edge_progress * side_length,
                -self.size,
            ));
        }

        let shape = PathShape::convex_polygon(
            points.into_iter().map(offset).collect(),
            self.fill,
            PathStroke::NONE,
        );
        painter.add(shape);
    }
}
