use crate::game::GamePauseState;
use crate::ui::commons::ShowGameUi;
use crate::ui::font_loader::AppFonts;
use crate::{define_asset_collection, prelude::*, AppState};
use bevy::{prelude::*, render::render_resource::*};

use crate::ui::theme::{METRICS_BAR_BG_COLOR, METRICS_BAR_FG_COLOR, METRICS_BAR_TEXT_COLOR};

pub struct ProgressBarsPlugin;
impl Plugin for ProgressBarsPlugin {
    fn build(&self, app: &mut App) {
        app.add_plugins(UiMaterialPlugin::<ProgressBarMaterial>::default());

        // we can add this in every gamemode, because unless someone adds bars to it
        // it wont be visible
        // (Also most gamemodes have progressbar support)
        app.add_systems(OnEnter(AppState::InGame), spawn_progressbars_container);

        app.add_systems(OnEnter(GamePauseState::Paused), hide_bars);
        app.add_systems(
            OnEnter(GamePauseState::Running),
            show_bars.run_if(in_state(ShowGameUi::On)),
        );

        app.add_systems(OnEnter(ShowGameUi::Off), hide_bars);
        app.add_systems(
            OnEnter(ShowGameUi::On),
            show_bars.run_if(in_state(GamePauseState::Running)),
        );

        app.add_systems(
            Update,
            (
                spawn_bar_layout,
                update_progressbars_progress,
                update_progressbars_text,
                update_progressbars_colors,
            )
                .run_if(in_state(AppState::InGame)),
        );

        app.register_asset_collection::<ProgressBarMaterials>();
    }
}

#[derive(AsBindGroup, Asset, TypePath, Debug, Clone)]
pub struct ProgressBarMaterial {
    /// Color multiplied with the image
    #[uniform(0)]
    color: Vec4,
    /// Represents how much of the image is visible
    /// Goes from 0 to 1
    /// A `Vec4` is used here because Bevy with webgl2 requires that uniforms are 16-byte aligned but only the first component is read.
    #[uniform(1)]
    progress: Vec4,
}

define_asset_collection!(
    ProgressBarMaterials,
    !bar : Shader = "shaders/progress_bar.wgsl",
);

impl UiMaterial for ProgressBarMaterial {
    fn fragment_shader() -> ShaderRef {
        "shaders/progress_bar.wgsl".into()
    }
}

/// top/left offset of the progressbars
const BAR_GROUP_MARGIN: f32 = 8.0;
/// Text size of the progressbar text
/// is scaled internally by bevy
const BAR_TEXT_SIZE: f32 = 14.0;
/// scale factor used to determine the bar height
/// multiplied with the scaled font size
const BAR_HEIGHT_FACTOR: f32 = 1.8;
/// scale factor used to determine the icon height
/// multiplied with the scaled font size
const BAR_ICON_HEIGHT_FACTOR: f32 = 1.2;
/// Horizontal bar padding in px
const BAR_HORIZONTAL_PADDING: f32 = 8.0;
/// Minimal bar width as estimated character count
/// Based on an average text character estimate
const BAR_MIN_WIDTH: f32 = 30.0;
/// Average character width estimate (guess)
const AVG_CHAR_RATIO: f32 = 2.0 / 3.0;
/// Maximum bar width in vmin
const BAR_MAX_WIDTH: f32 = 90.0;

pub fn spawn_progressbars_container(mut commands: Commands) {
    commands.spawn((
        Node {
            display: Display::Flex,
            flex_direction: FlexDirection::Column,
            position_type: PositionType::Absolute,
            top: Val::Px(BAR_GROUP_MARGIN),
            left: Val::Px(BAR_GROUP_MARGIN),
            row_gap: Val::Px(4.0),
            ..default()
        },
        ProgressBarContainer,
        StateScoped(AppState::InGame),
        Visibility::Visible,
    ));
}

fn hide_bars(mut containers: Query<&mut Visibility, With<ProgressBarContainer>>) {
    for mut vis in containers.iter_mut() {
        *vis = Visibility::Hidden;
    }
}
fn show_bars(mut containers: Query<&mut Visibility, With<ProgressBarContainer>>) {
    for mut vis in containers.iter_mut() {
        *vis = Visibility::Visible;
    }
}

#[derive(Component)]
pub struct ProgressBarContainer;

/// ProgressBar progress [0,1]
/// the other bar config options are in separate structs
/// to make it easier to use bevys change detection
#[derive(Component, Default, Deref, DerefMut)]
pub struct ProgressBar(pub f32);
/// Text displayed inside the bar
#[derive(Component, Deref, DerefMut)]
pub struct ProgressBarText(pub String);
/// Icon shown on the left of the bar
/// (optional)
/// can not be changed later
#[derive(Component, Deref, DerefMut)]
pub struct ProgressBarIcon(pub Handle<Image>);
#[derive(Component, Clone)]
pub struct ProgressBarColors {
    pub bg_color: Color,
    pub bar_color: Color,
    pub text_color: Color,
}
impl Default for ProgressBarColors {
    fn default() -> Self {
        Self {
            bg_color: METRICS_BAR_BG_COLOR.cast(),
            bar_color: METRICS_BAR_FG_COLOR.cast(),
            text_color: METRICS_BAR_TEXT_COLOR.cast(),
        }
    }
}

#[allow(clippy::type_complexity)]
fn spawn_bar_layout(
    new_bars: Query<
        (
            Entity,
            &ProgressBar,
            &ProgressBarText,
            Option<&ProgressBarIcon>,
            Option<&ProgressBarColors>,
        ),
        Added<ProgressBar>,
    >,
    mut commands: Commands,
    mut progressbar_materials: ResMut<Assets<ProgressBarMaterial>>,
    fonts: Res<AppFonts>,
) {
    let font = fonts.get_paragraph_font();

    let text_size = BAR_TEXT_SIZE;
    let bar_height = text_size * BAR_HEIGHT_FACTOR;
    let icon_size = text_size * BAR_ICON_HEIGHT_FACTOR;

    for (entity, progress, text, icon, colors) in new_bars.iter() {
        let theme = colors.cloned().unwrap_or(ProgressBarColors::default());
        commands
            .get_entity(entity)
            .expect("The Entity was obtained from the query, it should exist")
            .insert((
                Node {
                    position_type: PositionType::Relative,
                    display: Display::Flex,
                    // the icon is placed after the text in the node tree
                    // because it is optional, whereas the text isn't
                    // this allows using child indexes to access the bar-core and text without checking
                    // and checked icon access.
                    // We have to flip the flex direction for the icon to appear left of the text
                    flex_direction: FlexDirection::RowReverse,
                    align_items: AlignItems::Center,
                    // enforce left-aligned layout
                    justify_content: JustifyContent::Start,
                    overflow: Overflow::clip(),
                    overflow_clip_margin: OverflowClipMargin::padding_box(),
                    padding: UiRect::horizontal(Val::Px(BAR_HORIZONTAL_PADDING)),
                    height: Val::Px(bar_height),
                    min_width: Val::Px(BAR_MIN_WIDTH * AVG_CHAR_RATIO * BAR_TEXT_SIZE + icon_size),
                    max_width: Val::VMin(BAR_MAX_WIDTH),
                    column_gap: Val::Px(BAR_HORIZONTAL_PADDING / 2.0),
                    ..default()
                },
                // Hardcoded because the shader also uses a hardcoded 50%
                BorderRadius::all(Val::Percent(50.0)),
                BackgroundColor(theme.bg_color),
            ))
            .with_children(|builder| {
                // the progress bar itself
                builder.spawn((
                    Node {
                        position_type: PositionType::Absolute,
                        top: Val::Px(0.0),
                        left: Val::Px(0.0),
                        height: Val::Percent(100.0),
                        width: Val::Percent(100.0),
                        display: Display::Flex,
                        ..default()
                    },
                    MaterialNode(progressbar_materials.add(ProgressBarMaterial {
                        color: theme.bar_color.to_linear().to_vec4(),
                        progress: Vec4::splat(progress.clamp(0.0, 1.0)),
                    })),
                    ProgressBarIndicatorMarker,
                ));

                builder.spawn((
                    Text::new(text.0.clone()),
                    TextFont {
                        font: font.clone(),
                        font_size: text_size,
                        ..default()
                    },
                    ProgressBarTextMarker,
                    TextColor(theme.text_color),
                ));

                if let Some(ProgressBarIcon(handle)) = icon {
                    builder.spawn((
                        ImageNode {
                            image: handle.clone_weak(),
                            color: theme.text_color,
                            ..default()
                        },
                        Node {
                            // ensure the icon is square
                            aspect_ratio: Some(1.0),
                            height: Val::Px(icon_size),
                            ..default()
                        },
                    ));
                }
            });
    }
}

/// Marker struct to make querying for the progress indicator easier
#[derive(Component)]
pub struct ProgressBarIndicatorMarker;
/// update the progress indicator if the bar config changes
pub fn update_progressbars_progress(
    changed: Query<(&ProgressBar, &Children), Changed<ProgressBar>>,
    bars: Query<&MaterialNode<ProgressBarMaterial>, With<ProgressBarIndicatorMarker>>,
    mut materials: ResMut<Assets<ProgressBarMaterial>>,
) {
    for (conf, children) in changed.iter() {
        if children.len() < 2 {
            // invalid node configuration
            warn!("Invalid progressbar node configuration");
            continue;
        }
        let progress_ent = children.first().unwrap();
        materials
            .get_mut(
                &bars
                    .get(*progress_ent)
                    .expect("The progress indicator is the first child.")
                    .0,
            )
            .expect("A custom material has to be associated with the bar")
            .progress
            .x = conf.0.clamp(0.0, 1.0);
    }
}

/// Marker struct to make querying for the text easier
#[derive(Component)]
pub struct ProgressBarTextMarker;
pub fn update_progressbars_text(
    changed: Query<(&ProgressBarText, &Children), Changed<ProgressBarText>>,
    mut texts: Query<&mut Text, With<ProgressBarTextMarker>>,
) {
    for (conf, children) in changed.iter() {
        if children.len() < 2 {
            // invalid node configuration
            warn!("Invalid progressbar node configuration");
            continue;
        }
        let text_ent = children.get(1).unwrap();
        texts
            .get_mut(*text_ent)
            .expect("The text is the second child.")
            .0 = conf.0.clone();
    }
}

#[allow(clippy::type_complexity)]
pub fn update_progressbars_colors(
    mut changed: Query<
        (&ProgressBarColors, &Children, &mut BackgroundColor),
        (
            Changed<ProgressBarColors>,
            Without<ProgressBarIndicatorMarker>,
        ),
    >,
    bars: Query<
        &MaterialNode<ProgressBarMaterial>,
        (With<ProgressBarIndicatorMarker>, Without<ProgressBarColors>),
    >,
    mut materials: ResMut<Assets<ProgressBarMaterial>>,
    mut texts: Query<&mut TextColor, With<ProgressBarTextMarker>>,
    mut icons: Query<&mut ImageNode>,
) {
    for (conf, children, mut bg) in changed.iter_mut() {
        bg.0 = conf.bg_color;

        if children.len() < 2 {
            // invalid node configuration
            warn!("Invalid progressbar node configuration");
            continue;
        }
        let progress_ent = children.first().unwrap();
        materials
            .get_mut(
                &bars
                    .get(*progress_ent)
                    .expect("The progress indicator is the first child.")
                    .0,
            )
            .expect("The material has been associated to the bar when spawning it")
            .color = conf.bar_color.to_linear().to_vec4();
        let text_ent = children.get(1).unwrap();
        **texts
            .get_mut(*text_ent)
            .expect("The text is the second child.") = conf.text_color;
        if let Some(icon_child) = children.get(3) {
            if let Ok(mut icon) = icons.get_mut(*icon_child) {
                icon.color = conf.text_color;
            }
        }
    }
}
