use bevy::prelude::*;
use bevy_egui::{
    egui::{self},
    EguiContexts,
};

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        tiles::{
            assets::TilePreviews,
            sound_tile::{sound::TileDecoder, ChangeSoundTileData, SoundTileData, Wave, Waveform},
            TileType,
        },
    },
    i18n::{Localization, Translate},
    prelude::WithTranslationID,
    settings::ActiveSettingsBank,
};

use super::{popup_window, tile_info_display, TileInfoWindow};

pub fn sound_tile_clicked(
    mut tile_click_event: EventReader<TileClickEvent>,
    sound_tiles: Query<&SoundTileData>,
    map: Res<Map>,
    mut info_window: ResMut<TileInfoWindow>,
) {
    if tile_click_event.is_empty() {
        return;
    }

    let click = tile_click_event.read().last().unwrap();

    if let Some(Ok(_)) = map.get(click.tile_pos).map(|e| sound_tiles.get(e)) {
        info_window.open = true;
        info_window.selected_tile = Some(click.tile_pos);
    }
}

#[allow(clippy::too_many_arguments)]
pub fn render_sound_tile_popup(
    mut contexts: EguiContexts,
    sound_tiles: Query<&SoundTileData>,
    map: Res<Map>,
    mut tile_window: ResMut<TileInfoWindow>,
    localization: Res<Localization>,
    previews: Res<TilePreviews>,
    settings: Res<ActiveSettingsBank>,
    mut ev_configure: EventWriter<ChangeSoundTileData>,
) {
    // no sound tile selected -> don't render Window
    if !tile_window.open {
        return;
    }

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let Some(coord) = tile_window.selected_tile else {
        return;
    };

    // guaranteed to be Some because it is checked before
    let Some(Ok(data)) = map.get(coord).map(|e| sound_tiles.get(e)) else {
        return;
    };

    popup_window(
        egui,
        &mut tile_window.open,
        &localization.translate("sound_tile-popup"),
        |ui| {
            tile_info_display(
                ui,
                &TileType::Sound,
                &coord,
                &previews,
                &localization,
                &settings,
            );

            ui.separator();

            // settings
            let mut conf = ChangeSoundTileData {
                coord,
                data: data.clone(),
            };

            let mut has_changes = false;

            egui::Grid::new("sound_tile-popup-conf").show(ui, |ui| {
                ui.label(localization.translate("sound_tile-conf-enabled"));
                let enabled = ui.checkbox(&mut conf.data.enabled, "");
                has_changes |= enabled.changed();
                ui.end_row();

                ui.label(localization.translate("sound_tile-conf-amplify"));
                let amplify = ui.add(
                    egui::DragValue::new(&mut conf.data.amplify)
                        .speed(0.01)
                        .range(0.0..=1.0),
                );
                has_changes |= amplify.changed();
                ui.end_row();
            });

            if ui
                .button(localization.translate("sound_tile-popup-new_wave"))
                .clicked()
            {
                conf.data.waves.push(Wave::default());
                has_changes = true;
            }

            let mut delete_task = None;

            for (i, wave) in conf.data.waves.iter_mut().enumerate() {
                ui.separator();
                egui::Grid::new(format!("sound_tile-popup-conf-wave-{}", i)).show(ui, |ui| {
                    ui.label(localization.translate("sound_tile-conf-enabled"));
                    let enabled = ui.checkbox(&mut wave.enabled, "");
                    has_changes |= enabled.changed();
                    ui.end_row();

                    ui.label(localization.translate("sound_tile-conf-frequency"));
                    let freq = ui.add(
                        egui::DragValue::new(&mut wave.frequency)
                            .speed(1.0)
                            .range(0.0..=TileDecoder::SAMPLE_RATE as f32),
                    );
                    has_changes |= freq.changed();
                    ui.end_row();

                    ui.label(localization.translate("sound_tile-conf-waveform"));
                    egui::ComboBox::from_id_salt(format!("sound_tile-popup-wave-{}-waveform", i))
                        .selected_text(localization.translate(wave.waveform.get_translation_id()))
                        .show_ui(ui, |ui| {
                            for waveform in Waveform::list() {
                                if ui
                                    .selectable_value(
                                        &mut wave.waveform,
                                        waveform,
                                        localization.translate(waveform.get_translation_id()),
                                    )
                                    .changed()
                                {
                                    has_changes = true;
                                };
                            }
                        });
                    ui.end_row();

                    ui.label(localization.translate("sound_tile-conf-offset"));

                    // offset enable
                    let mut offset_state = wave.offset.is_some();
                    let offset_checkbox = ui.checkbox(
                        &mut offset_state,
                        localization.translate("sound_tile-conf-offset-activated"),
                    );
                    if offset_checkbox.changed() {
                        if offset_state {
                            wave.offset = Some(0.0);
                        } else {
                            wave.offset = None;
                        }
                        has_changes = true;
                    }

                    // offset box
                    if let Some(offset) = &mut wave.offset {
                        let wave_len = 1.0 / wave.frequency as f64;
                        let offset = ui.add(
                            egui::DragValue::new(offset)
                                .speed(wave_len / 1000.0)
                                .range(-wave_len..=wave_len)
                                .custom_formatter(|val, _| format!("{:.1}", val / wave_len * 100.0))
                                .custom_parser(|s| Some(s.parse::<f64>().ok()? / 100.0 * wave_len)),
                        );
                        has_changes |= offset.changed();
                    }

                    ui.end_row();

                    ui.label(localization.translate("sound_tile-conf-amplify"));
                    let amplify = ui.add(
                        egui::DragValue::new(&mut wave.amplify)
                            .speed(0.01)
                            .range(0.0..=f32::INFINITY),
                    );
                    has_changes |= amplify.changed();
                    ui.end_row();
                });

                if ui
                    .button(localization.translate("sound_tile-popup-delete_wave"))
                    .clicked()
                {
                    delete_task = Some(i);
                }
            }

            if let Some(index) = delete_task {
                conf.data.waves.remove(index);
                has_changes = true;
            }

            if has_changes {
                ev_configure.send(conf);
            }
        },
    );
}
