use bevy::prelude::*;
use bevy_egui::{
    egui::{self},
    EguiContexts,
};

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        tiles::{
            assets::TilePreviews,
            rgb_tile::{ChangeRgbTileColor, RgbTileData},
            TileType,
        },
    },
    i18n::{Localization, Translate},
    settings::ActiveSettingsBank,
};

use super::{popup_window, tile_info_display, TileInfoWindow};

pub fn rgb_tile_clicked(
    mut tile_click_event: EventReader<TileClickEvent>,
    rgb_tiles: Query<&RgbTileData>,
    map: Res<Map>,
    mut info_window: ResMut<TileInfoWindow>,
) {
    if tile_click_event.is_empty() {
        return;
    }

    let click = tile_click_event.read().last().unwrap();

    if let Some(Ok(_)) = map.get(click.tile_pos).map(|e| rgb_tiles.get(e)) {
        info_window.open = true;
        info_window.selected_tile = Some(click.tile_pos);
    }
}

#[allow(clippy::too_many_arguments)]
pub fn render_rgb_tile_popup(
    mut contexts: EguiContexts,
    rgb_tiles: Query<&RgbTileData>,
    map: Res<Map>,
    mut ev_configure: EventWriter<ChangeRgbTileColor>,
    mut tile_window: ResMut<TileInfoWindow>,
    localization: Res<Localization>,
    previews: Res<TilePreviews>,
    settings: Res<ActiveSettingsBank>,
) {
    // no rgb tile selected -> don't render Window
    if !tile_window.open {
        return;
    }

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let Some(coord) = tile_window.selected_tile else {
        return;
    };

    // guaranteed to be Some because it is checked before
    let Some(Ok(color)) = map.get(coord).map(|e| rgb_tiles.get(e)) else {
        return;
    };

    popup_window(
        egui,
        &mut tile_window.open,
        &localization.translate("rgb_tile-popup"),
        |ui| {
            tile_info_display(
                ui,
                &TileType::Rgb,
                &coord,
                &previews,
                &localization,
                &settings,
            );

            ui.separator();

            // settings
            let mut conf = ChangeRgbTileColor {
                coord,
                rgb: RgbTileData {
                    r: color.r,
                    g: color.g,
                    b: color.b,
                },
            };

            let r_slider = ui.add(
                egui::Slider::new(&mut conf.rgb.r, 0..=255)
                    .text(localization.translate("value-red"))
                    .step_by(1.0),
            );
            let g_slider = ui.add(
                egui::Slider::new(&mut conf.rgb.g, 0..=255)
                    .text(localization.translate("value-green"))
                    .step_by(1.0),
            );
            let b_slider = ui.add(
                egui::Slider::new(&mut conf.rgb.b, 0..=255)
                    .text(localization.translate("value-blue"))
                    .step_by(1.0),
            );

            if r_slider.changed() || g_slider.changed() || b_slider.changed() {
                ev_configure.send(conf);
            }
        },
    );
}
