use bevy::prelude::*;
use bevy_egui::{
    egui::{self},
    EguiContexts,
};

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        metrics::MetricsRate,
        resources::{marketplace::SellKeepRatio, ConfigureMarketReq, MarketInfo},
        tiles::{assets::TilePreviews, TileType},
    },
    i18n::{Localization, Translate},
    settings::ActiveSettingsBank,
};

use super::{popup_window, tile_info_display, TileInfoWindow};

pub fn market_place_clicked(
    mut tile_click_event: EventReader<TileClickEvent>,
    marketplaces: Query<&MarketInfo>,
    map: Res<Map>,
    mut market_window: ResMut<TileInfoWindow>,
) {
    if tile_click_event.is_empty() {
        return;
    }

    let click = tile_click_event.read().last().unwrap();

    if let Some(Ok(_)) = map.get(click.tile_pos).map(|e| marketplaces.get(e)) {
        market_window.open = true;
        market_window.selected_tile = Some(click.tile_pos);
    }
}

#[allow(clippy::too_many_arguments)]
pub fn render_market_popup(
    mut contexts: EguiContexts,
    marketplaces: Query<(&MetricsRate, &MarketInfo)>,
    map: Res<Map>,
    mut ev_configure_ratio: EventWriter<ConfigureMarketReq>,
    mut tile_window: ResMut<TileInfoWindow>,
    localization: Res<Localization>,
    previews: Res<TilePreviews>,
    settings: Res<ActiveSettingsBank>,
) {
    // no market selected -> don't render Window
    if !tile_window.open {
        return;
    }

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let Some(sel_market_coord) = tile_window.selected_tile else {
        return;
    };

    // guaranteed to be Some because it is checked before
    let Some(Ok((sel_market_rate, sel_market))) =
        map.get(sel_market_coord).map(|e| marketplaces.get(e))
    else {
        return;
    };

    popup_window(
        egui,
        &mut tile_window.open,
        &localization.translate("market-popup"),
        |ui| {
            tile_info_display(
                ui,
                &TileType::Marketplace,
                &sel_market_coord,
                &previews,
                &localization,
                &settings,
            );

            ui.separator();

            // settings
            egui::Grid::new("market-popup-info-stats")
                .num_columns(3)
                .spacing([40.0, 4.0])
                .striped(true)
                .show(ui, |ui| {
                    // total values
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-amount-total")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        (-sel_market.economy.food).max(0.0),
                        localization.translate("game-metrics-food")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        (-sel_market.economy.materials).max(0.0),
                        localization.translate("game-metrics-materials")
                    ));
                    ui.end_row();
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-money-total")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_market.economy.money,
                        localization.translate("game-metrics-money")
                    ));
                    ui.end_row();

                    // current values
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-amount-available")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_market.available.food,
                        localization.translate("game-metrics-food")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_market.available.materials,
                        localization.translate("game-metrics-materials")
                    ));
                    ui.end_row();
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-amount-selling")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        (-sel_market_rate.food).max(0.0),
                        localization.translate("game-metrics-food")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        (-sel_market_rate.materials).max(0.0),
                        localization.translate("game-metrics-materials")
                    ));
                    ui.end_row();
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-money-current")
                    ));
                    ui.label(format!(
                        "{:.2} {}",
                        sel_market_rate.money,
                        localization.translate("game-metrics-money")
                    ));
                    ui.end_row();
                    ui.label(format!(
                        "{}:",
                        localization.translate("market-popup-trade-efficiency")
                    ));
                    ui.label(format!("{:.0}%", sel_market.trade_efficiency * 100.0));
                });

            let mut slider = ConfigureMarketReq {
                coord: sel_market_coord,
                ratio: SellKeepRatio {
                    food: sel_market.trade.food,
                    materials: sel_market.trade.materials,
                },
            };

            let food_slider = ui.add(
                egui::Slider::new(&mut slider.ratio.food, 0.0..=1.)
                    .text(localization.translate("game-metrics-food"))
                    .step_by(0.01)
                    .custom_formatter(|n, _| format!("{:.0}%", n * 100.0))
                    .custom_parser(|s| s.parse::<f64>().map(|n| n / 100.0).ok()),
            );

            let material_slider = ui.add(
                egui::Slider::new(&mut slider.ratio.materials, 0.0..=1.0)
                    .text(localization.translate("game-metrics-materials"))
                    .step_by(0.01)
                    .custom_formatter(|n, _| format!("{:.0}%", n * 100.0))
                    .custom_parser(|s| s.parse::<f64>().map(|n| n / 100.0).ok()),
            );

            if food_slider.changed() || material_slider.changed() {
                ev_configure_ratio.send(slider);
            }
        },
    );
}
