use bevy::prelude::*;
use bevy_egui::{egui, EguiContexts};

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        tiles::{
            assets::TilePreviews,
            chameleon_tile::{ChameleonTileData, ChangeChameleonTile},
            TileAvailability, TileType,
        },
    },
    i18n::{Localization, Translate},
    settings::ActiveSettingsBank,
};

use super::{popup_window, tile_info_display, TileInfoWindow};

pub fn chameleon_tile_clicked(
    mut tile_click_event: EventReader<TileClickEvent>,
    rgb_tiles: Query<&ChameleonTileData>,
    map: Res<Map>,
    mut info_window: ResMut<TileInfoWindow>,
) {
    if tile_click_event.is_empty() {
        return;
    }

    let click = tile_click_event.read().last().unwrap();

    if let Some(Ok(_)) = map.get(click.tile_pos).map(|e| rgb_tiles.get(e)) {
        info_window.open = true;
        info_window.selected_tile = Some(click.tile_pos);
    }
}

#[allow(clippy::too_many_arguments)]
pub fn render_chameleon_tile_popup(
    mut contexts: EguiContexts,
    chameleon_tiles: Query<&ChameleonTileData>,
    map: Res<Map>,
    mut ev_configure: EventWriter<ChangeChameleonTile>,
    mut tile_window: ResMut<TileInfoWindow>,
    localization: Res<Localization>,
    previews: Res<TilePreviews>,
    settings: Res<ActiveSettingsBank>,
) {
    // no chameleon tile selected -> don't render Window
    if !tile_window.open {
        return;
    }

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let Some(coord) = tile_window.selected_tile else {
        return;
    };

    // guaranteed to be Some because it is checked before
    let Some(Ok(data)) = map.get(coord).map(|e| chameleon_tiles.get(e)) else {
        return;
    };

    popup_window(
        egui,
        &mut tile_window.open,
        &localization.translate("rgb_tile-popup"),
        |ui| {
            tile_info_display(
                ui,
                &TileType::Chameleon,
                &coord,
                &previews,
                &localization,
                &settings,
            );

            ui.separator();

            // settings
            let mut conf = ChangeChameleonTile {
                coord,
                data: data.clone(),
            };

            let mut model_change = false;
            ui.horizontal(|ui| {
                ui.label(localization.translate("chameleon_tile-popup-mode"));
                egui::ComboBox::from_id_salt("chameleon-tile-popup-model")
                    // NOTE: use the tile name here, because we modify the tile in the world and not in the hand
                    .selected_text(
                        localization.translate(conf.data.model.get_tile_translation_id()),
                    )
                    .show_ui(ui, |ui| {
                        for tile in TileType::list()
                            .into_iter()
                            .filter(|tile| tile.get_availability() != TileAvailability::Creative)
                        {
                            if ui
                                .selectable_value(
                                    &mut conf.data.model,
                                    tile,
                                    localization.translate(tile.get_tile_translation_id()),
                                )
                                .changed()
                            {
                                model_change = true;
                            }
                        }
                    });
            });

            ui.label(localization.translate("chameleon_tile-popup-text"));

            let text_resp = ui.add(
                egui::TextEdit::multiline(&mut conf.data.text)
                    .code_editor()
                    .hint_text(localization.translate("chameleon_tile-popup-text-hint")),
            );

            if model_change || text_resp.changed() {
                ev_configure.send(conf);
            }
        },
    );
}
