use crate::{define_asset_collection, prelude::*};
use bevy::prelude::*;
use bevy_egui::egui::{self, load::SizedTexture};

pub mod campaign;
pub mod creative_inventory;
#[cfg(debug_assertions)]
pub mod debug_placer;
pub mod game_export;
mod game_speed;
pub mod hand;
mod inventory;
mod layout;
pub mod metrics;
pub mod pause;
pub mod popup;
pub mod progress_bars;
mod progress_rect;
pub mod rounds;
pub mod settings;

use campaign::CampaignUiPlugin;
use creative_inventory::CreativeInventoryUiPlugin;
#[cfg(debug_assertions)]
use debug_placer::DebugTilePlacerUiPlugin;
use game_export::GameExportUi;
use game_speed::GameSpeedPlugin;
use hand::HandUiPlugin;
use layout::LayoutManagerPlugin;
use metrics::{MetricsDisplayPlugin, StatisticsViewState};
use pause::PauseUiPlugin;
use popup::TilePopupPlugin;
use progress_bars::ProgressBarsPlugin;
use rounds::RoundsUiPlugin;
use settings::InGameSettingsPlugin;

use super::theme::{
    BUTTON_COLOR, INNER_MARGIN, INV_SLOT_BG, INV_SLOT_RADIUS, INV_SLOT_STROKE,
    INV_SLOT_STROKE_HOVER,
};

define_asset_collection!(
    CommonInventoryAssets,
    !menu_open : Image = "icons/menuGrid.png",
);

pub struct InGameUiPlugin;
impl Plugin for InGameUiPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<CommonInventoryAssets>();
        app.add_plugins(LayoutManagerPlugin);
        app.add_plugins(PauseUiPlugin);
        app.add_sub_state::<StatisticsViewState>();
        app.add_plugins(InGameSettingsPlugin);
        app.add_plugins(CampaignUiPlugin);
        app.add_plugins(GameSpeedPlugin);
        app.add_plugins(CreativeInventoryUiPlugin);
        #[cfg(debug_assertions)]
        app.add_plugins(DebugTilePlacerUiPlugin);
        app.add_plugins(HandUiPlugin);
        app.add_plugins(MetricsDisplayPlugin);
        app.add_plugins(GameExportUi);
        app.add_plugins(RoundsUiPlugin);
        app.add_plugins(TilePopupPlugin);
        app.add_plugins(ProgressBarsPlugin);
    }
}

pub struct IconButtonCard {
    /// size of the icon
    icon_size: f32,
    /// icon texture to show
    texture: egui::TextureId,
}
impl IconButtonCard {
    pub fn get_size(&self) -> f32 {
        self.icon_size + 2.0 * INNER_MARGIN
    }
}
impl egui::Widget for IconButtonCard {
    fn ui(self, ui: &mut egui::Ui) -> egui::Response {
        // manually create the frame, to make it clickable
        let mut prep = egui::Frame::group(ui.style())
            .corner_radius(INV_SLOT_RADIUS)
            .fill(INV_SLOT_BG)
            .inner_margin(egui::Margin::same(INNER_MARGIN as i8))
            .stroke(egui::Stroke::new(2.0, INV_SLOT_STROKE))
            .begin(ui);
        // manually add the content
        {
            let ui = &mut prep.content_ui;
            ui.style_mut().interaction.selectable_labels = false;
            ui.add(
                egui::widgets::Image::new(SizedTexture::new(
                    self.texture,
                    egui::Vec2::new(self.icon_size, self.icon_size),
                ))
                .tint(BUTTON_COLOR),
            );
        };
        let content_with_margin =
            prep.content_ui.min_rect() + (prep.frame.inner_margin + prep.frame.outer_margin);
        let alloc = ui.allocate_rect(content_with_margin, egui::Sense::click());
        if alloc.hovered() {
            prep.frame.fill = INV_SLOT_STROKE_HOVER;
        }
        prep.paint(ui);
        alloc
    }
}
