use std::time::Duration;

use bevy::{ecs::system::SystemParam, prelude::*};
use bevy_egui::{
    egui::{self, Vec2b},
    EguiContexts,
};
use egui_plot::{log_grid_spacer, Legend, Line, Plot, PlotBounds, PlotMemory, PlotTransform};

use crate::{
    define_asset_collection,
    game::{
        controls::KeyboardEvent,
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
                CampaignGroups,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
            GameMode,
        },
        metrics::{Metrics, MetricsRate, TargetMetrics},
        round::StatisticsData,
        time::{Game, GameTime},
        GamePauseState, GameSimSet,
    },
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        in_game::{
            progress_bars::{
                spawn_progressbars_container, update_progressbars_colors,
                update_progressbars_progress, update_progressbars_text,
            },
            rounds::spawn_round_bar,
        },
        is_typing_egui,
        theme::{
            BACKGROUND_ALPHA, BACKGROUND_COLOR, BIG_TEXT_SIZE, BUTTON_COLOR, BUTTON_TEXT_COLOR,
            MARGIN_SIZE, METRICS_BAR_FG_COLOR, METRICS_BAR_FG_COLOR_COMPLETED, OUTER_MARGIN,
            PARAGRAPH_FONT_NAME, PLOT_FOOD_COLOR, PLOT_MATERIALS_COLOR, PLOT_MAX_HEIGHT,
            PLOT_MAX_WIDTH, PLOT_MONEY_COLOR, PLOT_STROKE_WIDTH, PLOT_TARGET_COLOR,
            PRIMARY_BUTTON_SIZE, PRIMARY_BUTTON_TEXT_SIZE, TITLE_FONT_NAME, TITLE_TEXT_COLOR,
        },
        UiClickEvent,
    },
};

use super::{
    layout::{min_f32, LayoutManager},
    pause::{ui_pause_button, PauseUiState},
    progress_bars::{
        ProgressBar, ProgressBarColors, ProgressBarContainer, ProgressBarIcon, ProgressBarText,
    },
    IconButtonCard,
};

macro_rules! define_metrics_display_plugin {
    ($({
        // gamemode in which to use bars
        gamemode: $gamemode:expr,
        // the state in which the top-left progress bars are supposed to be shown
        // and where the stats screen and buttons can be accessed
        state: $state:expr,
        // Additional spectate state
        $(spectate: $spectate:expr,)?
   }),*) => {
        /// Plugin that adds support for the metrics bar in the top left
        /// and the statistics screen
        pub struct MetricsDisplayPlugin;
        impl Plugin for MetricsDisplayPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                app.register_asset_collection::<ResourceIcons>();

                $(
                    app.add_systems(OnEnter($gamemode), spawn_bars.after(spawn_progressbars_container).after(spawn_round_bar));
                )*

                app.add_systems(
                    Update,
                    update_bars
                        .before(update_progressbars_text)
                        .before(update_progressbars_colors)
                        .before(update_progressbars_progress)
                        .after(GameSimSet::Simulation)
                        .run_if(in_state(ShowGameUi::On))
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(never()$(.or(in_state($state)))*));

                app.add_systems(
                    Update,
                    ui_open_stats_button
                        .run_if(never()$(.or(in_state($state))$(.or(in_state($spectate)))?)*)
                        .run_if(in_state(ShowGameUi::On))
                        .run_if(in_state(GamePauseState::Running))
                        .after(ui_pause_button)
                );
                app.register_asset_collection::<StatsIcon>();
                app.add_systems(
                    Update,
                    ui_show_stats
                        .run_if(never()$(.or(in_state($state))$(.or(in_state($spectate)))?)*)
                        .run_if(
                            in_state(StatisticsViewState::FromGame)
                                .or(in_state(StatisticsViewState::FromPaused)),
                        ),
                );
                app.init_resource::<StatsUiContext>();
                app.add_systems(
                    Update,
                    toggle_statistics_view
                    // open the statistics view from ingame
                        .run_if(
                            in_state(GamePauseState::Running)
                            // close the stats view
                                .or(not(in_state(StatisticsViewState::Hidden)))
                            // open the stats view from the pause screen
                                .or(in_state(PauseUiState::Shown)),
                        )
                        .run_if(never()$(.or(in_state($state))$(.or(in_state($spectate)))?)*)
                );
                app.add_systems(
                    OnExit(StatisticsViewState::Hidden),
                    reset_plot_view
                        .run_if(never()$(.or(in_state($state))$(.or(in_state($spectate)))?)*)
                );
            }
        }
    }
}
define_metrics_display_plugin![
    {
        gamemode: GameMode::Challenge,
        state: ChallengeGameState::InRound,
        spectate: ChallengeGameState::Spectate,
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        state: CampaignChallengeState::InRound,
        spectate: CampaignChallengeState::Spectate,
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        state: CampaignWettbewerbState::InRound,
        spectate: CampaignWettbewerbState::Spectate,
    },
    {
        gamemode: GameMode::Zen,
        state: ZenGameState::InRound,
    }
];

/// Marker struct for Food progress bar
#[derive(Component)]
struct FoodProgressBar;
/// Marker struct for Materials progress bar
#[derive(Component)]
struct MaterialsProgressBar;
/// Marker struct for Money progress bar
#[derive(Component)]
struct MoneyProgressBar;

define_asset_collection!(
    ResourceIcons,
    !food : Image = "icons/resources/food.png",
    !money: Image = "icons/resources/money.png",
    !materials : Image = "icons/resources/materials.png",
);

fn spawn_bars(
    mut commands: Commands,
    containers: Query<Entity, With<ProgressBarContainer>>,
    localization: Res<Localization>,
    icons: Res<ResourceIcons>,
) {
    // we wouldn't know where to spawn the bars,
    // if there was more than one container
    let Ok(container) = containers.get_single() else {
        return;
    };

    commands
        .get_entity(container)
        .expect("We obtained the entity from the query, so it should exist")
        .with_children(|builder| {
            builder.spawn((
                ProgressBar(0.0),
                ProgressBarText(format!(
                    "{} 0/0 (0/s)",
                    localization.translate("game-metrics-food")
                )),
                FoodProgressBar,
                ProgressBarColors::default(),
                ProgressBarIcon(icons.food.clone_weak()),
            ));
            builder.spawn((
                ProgressBar(0.0),
                ProgressBarText(format!(
                    "{} 0/0 (0/s)",
                    localization.translate("game-metrics-materials")
                )),
                MaterialsProgressBar,
                ProgressBarColors::default(),
                ProgressBarIcon(icons.materials.clone_weak()),
            ));
            builder.spawn((
                ProgressBar(0.0),
                ProgressBarText(format!(
                    "{} 0/0 (0/s)",
                    localization.translate("game-metrics-money")
                )),
                MoneyProgressBar,
                ProgressBarColors::default(),
                ProgressBarIcon(icons.money.clone_weak()),
            ));
        });
}

fn util_update_text(
    id: &str,
    progress: (f32, f32),
    rate: f32,
    bar_data: (
        &mut ProgressBar,
        &mut ProgressBarText,
        &mut ProgressBarColors,
    ),
    localization: &Localization,
) {
    let (has, needs) = progress;
    let (progress, text, colors) = bar_data;
    **progress = has / needs;
    **text = format!(
        "{}: {}/{} ({}/s)",
        localization.translate(id),
        has.as_human_readable(),
        needs.as_human_readable(),
        rate.as_human_readable(),
    );
    colors.bar_color = if **progress > 1.0 {
        METRICS_BAR_FG_COLOR_COMPLETED.cast()
    } else {
        METRICS_BAR_FG_COLOR.cast()
    }
}

#[allow(clippy::too_many_arguments)]
#[allow(clippy::type_complexity)]
fn update_bars(
    // progress bar config
    mut bars: ParamSet<(
        Query<
            (
                &mut ProgressBar,
                &mut ProgressBarText,
                &mut ProgressBarColors,
            ),
            With<FoodProgressBar>,
        >,
        Query<
            (
                &mut ProgressBar,
                &mut ProgressBarText,
                &mut ProgressBarColors,
            ),
            With<MaterialsProgressBar>,
        >,
        Query<
            (
                &mut ProgressBar,
                &mut ProgressBarText,
                &mut ProgressBarColors,
            ),
            With<MoneyProgressBar>,
        >,
    )>,

    // data sources
    metrics: Res<Metrics>,
    metrics_rate: Res<MetricsRate>,
    target_metrics: Res<TargetMetrics>,
    localization: Res<Localization>,
    gtime: Res<Time<Game>>,
) {
    let sub_secs = gtime.sub_secs();

    for (mut progress, mut text, mut colors) in bars.p0().iter_mut() {
        util_update_text(
            "game-metrics-food",
            (
                metrics.food + metrics_rate.food * sub_secs,
                target_metrics.food,
            ),
            metrics_rate.food,
            (&mut *progress, &mut *text, &mut *colors),
            &localization,
        );
    }
    for (mut progress, mut text, mut colors) in bars.p1().iter_mut() {
        util_update_text(
            "game-metrics-materials",
            (
                metrics.materials + metrics_rate.materials * sub_secs,
                target_metrics.materials,
            ),
            metrics_rate.materials,
            (&mut *progress, &mut *text, &mut *colors),
            &localization,
        );
    }
    for (mut progress, mut text, mut colors) in bars.p2().iter_mut() {
        util_update_text(
            "game-metrics-money",
            (
                metrics.money + metrics_rate.money * sub_secs,
                target_metrics.money,
            ),
            metrics_rate.money,
            (&mut *progress, &mut *text, &mut *colors),
            &localization,
        );
    }
}

/// draws the open statistics view button
/// next to the pause button
pub fn ui_open_stats_button(
    windows: Query<&Window>,
    mut layout: ResMut<LayoutManager>,
    mut contexts: EguiContexts,
    mut next_stats: ResMut<NextState<StatisticsViewState>>,
    mut next_pause: ResMut<NextState<GamePauseState>>,
    mut click_event: EventWriter<UiClickEvent>,
    icon: Res<StatsIcon>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.width() == 0.0 || window.height() == 0.0 {
        return;
    }

    let icon_texture = contexts.add_image(icon.icon.clone_weak());
    let icon_size = window.height() * BIG_TEXT_SIZE / 4.0;

    let icon = IconButtonCard {
        icon_size,
        texture: icon_texture,
    };

    let cell = if let Some(cell) = layout.allocate_margin(
        super::layout::LayoutOrigin::RightTop,
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) {
        cell
    } else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    egui::Area::new("open-stats-button".into())
        .pivot(egui::Align2::RIGHT_TOP)
        .fixed_pos(cell.rect.right_top())
        .show(egui, |ui| {
            if ui.add(icon).clicked() {
                click_event.send(UiClickEvent);
                next_stats.set(StatisticsViewState::FromGame);
                next_pause.set(GamePauseState::Paused);
            };
        });
}

#[derive(SubStates, Default, Clone, Hash, PartialEq, Eq, Debug)]
#[source(GamePauseState = GamePauseState::Paused)]
pub enum StatisticsViewState {
    /// statistics screen is not visible but game is paused
    #[default]
    Hidden,
    /// the stats screen was triggert from in-game
    /// autopause on enter
    /// autocontinue on exit
    FromGame,
    /// the statisticsview was opened from the pause screen
    FromPaused,
}

const PLOT_ID: &str = "statistics-plot";

/// reset previous plot movement & zoom
fn reset_plot_view(mut contexts: EguiContexts) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    if let Some(mut mem) = PlotMemory::load(egui, PLOT_ID.into()) {
        mem.auto_bounds.x = true;
        mem.auto_bounds.y = true;
        mem.set_transform(PlotTransform::new(
            *mem.transform().frame(),
            PlotBounds::NOTHING,
            Vec2b::FALSE,
        ));
        mem.store(egui, PLOT_ID.into());
    }
}

#[derive(Default, PartialEq)]
enum StatsGraphType {
    /// plot that displays the metrics and the target metrics
    /// over time
    #[default]
    Metrics,
    /// plot that displays the metrics rate over time
    MetricsRate,
}
#[derive(Default, Resource)]
struct StatsUiContext {
    /// which graph is supposed to be displayed
    page: StatsGraphType,
}

define_asset_collection!(
    StatsIcon,
    !icon : Image = "icons/buttons/stats.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

/// display the statistics overview
/// returns to game if triggert using the ingame button
/// returns to pause screen if triggert from pause screen
#[allow(clippy::too_many_arguments)]
fn ui_show_stats(
    mut contexts: EguiContexts,
    windows: Query<&Window>,
    localization: Res<Localization>,
    mut stats_next: ResMut<NextState<StatisticsViewState>>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
    mut pauseui: ResMut<NextState<PauseUiState>>,
    mut click_event: EventWriter<UiClickEvent>,
    data: Res<StatisticsData>,
    stats: Res<State<StatisticsViewState>>,
    mut stats_ctx: ResMut<StatsUiContext>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same((window.y * MARGIN_SIZE) as i8),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    // center panel
    egui::CentralPanel::default()
        .frame(my_frame)
        // .show_separator_line(false) // doesn't work
        .show(egui, |_ui| {});

    let reset_plot = egui::Area::new("statistics-overlay".into())
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    egui::RichText::new(localization.translate("metrics-overview-statistics"))
                        .font(egui::FontId::new(
                            window.y * BIG_TEXT_SIZE,
                            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(TITLE_TEXT_COLOR),
                );

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                let points = data.metrics.len();

                let mut target_metrics_vec: Vec<[f64; 2]> = Vec::with_capacity(points);
                let mut metrics_food_vec: Vec<[f64; 2]> = Vec::with_capacity(points);
                let mut metrics_materials_vec: Vec<[f64; 2]> = Vec::with_capacity(points);
                let mut metrics_money_vec: Vec<[f64; 2]> = Vec::with_capacity(points);

                let mut max_y = 0.0;
                let mut max_x = Duration::from_millis(0);

                let lines = match stats_ctx.page {
                    StatsGraphType::Metrics => {
                        // append data of past rounds to the vecs
                        for (time, metrics, target) in data.metrics.iter() {
                            let x = time.as_secs_f64();
                            // all resources targets have the same growth so we only have to draw one line
                            target_metrics_vec.push([x, target.food as f64]);
                            metrics_food_vec.push([x, metrics.food as f64]);
                            metrics_materials_vec.push([x, metrics.materials as f64]);
                            metrics_money_vec.push([x, metrics.money as f64]);

                            if target.food as f64 > max_y {
                                max_y = target.food as f64;
                            }
                            if metrics.food as f64 > max_y {
                                max_y = metrics.food as f64;
                            }
                            if metrics.materials as f64 > max_y {
                                max_y = metrics.materials as f64;
                            }
                            if metrics.money as f64 > max_y {
                                max_y = metrics.money as f64;
                            }

                            if time > &max_x {
                                max_x = *time;
                            }
                        }

                        // create lines for all tracked values
                        let target_metrics = Line::new(target_metrics_vec)
                            .name(localization.translate("metrics-overview-target"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_TARGET_COLOR);
                        let metrics_food = Line::new(metrics_food_vec)
                            .name(localization.translate("game-metrics-food"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_FOOD_COLOR);
                        let metrics_materials = Line::new(metrics_materials_vec)
                            .name(localization.translate("game-metrics-materials"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_MATERIALS_COLOR);
                        let metrics_money = Line::new(metrics_money_vec)
                            .name(localization.translate("game-metrics-money"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_MONEY_COLOR);

                        vec![
                            target_metrics,
                            metrics_food,
                            metrics_materials,
                            metrics_money,
                        ]
                    }
                    StatsGraphType::MetricsRate => {
                        let mut metrics_rate_food_vec = Vec::with_capacity(data.metrics_rate.len());
                        let mut metrics_rate_materials_vec =
                            Vec::with_capacity(data.metrics_rate.len());
                        let mut metrics_rate_money_vec =
                            Vec::with_capacity(data.metrics_rate.len());

                        for (time, metrics_rate) in data.metrics_rate.iter() {
                            let x = time.as_secs_f64();
                            // all resources targets have the same growth so we only have to draw one line
                            metrics_rate_food_vec.push([x, metrics_rate.food as f64]);
                            metrics_rate_materials_vec.push([x, metrics_rate.materials as f64]);
                            metrics_rate_money_vec.push([x, metrics_rate.money as f64]);

                            if metrics_rate.food as f64 > max_y {
                                max_y = metrics_rate.food as f64;
                            }
                            if metrics_rate.materials as f64 > max_y {
                                max_y = metrics_rate.materials as f64;
                            }
                            if metrics_rate.money as f64 > max_y {
                                max_y = metrics_rate.money as f64;
                            }

                            if time > &max_x {
                                max_x = *time;
                            }
                        }

                        let metrics_rate_food = Line::new(metrics_rate_food_vec)
                            .name(localization.translate("game-metrics-food"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_FOOD_COLOR);
                        let metrics_rate_materials = Line::new(metrics_rate_materials_vec)
                            .name(localization.translate("game-metrics-materials"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_MATERIALS_COLOR);
                        let metrics_rate_money = Line::new(metrics_rate_money_vec)
                            .name(localization.translate("game-metrics-money"))
                            .width(PLOT_STROKE_WIDTH)
                            .color(PLOT_MONEY_COLOR);

                        vec![
                            metrics_rate_food,
                            metrics_rate_materials,
                            metrics_rate_money,
                        ]
                    }
                };

                // adjust the plot width to fit the display aspect ratio
                let (width, height) = if window.x >= window.y {
                    let height = window.y * PLOT_MAX_HEIGHT;
                    let width = height * window.x / window.y;
                    (min_f32(width, PLOT_MAX_WIDTH * window.x), height)
                } else {
                    let width = window.x * PLOT_MAX_WIDTH;
                    let height = width * window.y / window.x;
                    (width, min_f32(height, PLOT_MAX_HEIGHT * window.y))
                };

                // plot selector
                let reset_plot = ui.horizontal(|ui| {
                    let mut reset = false;
                    if ui
                        .selectable_value(
                            &mut stats_ctx.page,
                            StatsGraphType::Metrics,
                            localization.translate("metrics-overview-graph-metrics"),
                        )
                        .clicked()
                    {
                        reset = true;
                    }
                    if ui
                        .selectable_value(
                            &mut stats_ctx.page,
                            StatsGraphType::MetricsRate,
                            localization.translate("metrics-overview-graph-metrics_rate"),
                        )
                        .clicked()
                    {
                        reset = true;
                    }
                    reset
                });

                // show plot
                Plot::new(PLOT_ID)
                    .id(PLOT_ID.into())
                    // set the plot size
                    .height(height)
                    .width(width)
                    // make sure that all values are contained
                    // within the view by default
                    .include_x(0.0)
                    .include_x(max_x.as_secs_f64())
                    .include_y(0.0)
                    .include_y(max_y)
                    // show a legend with color and line name
                    .legend(Legend::default().position(egui_plot::Corner::RightBottom))
                    // make the plot transparent
                    .show_background(false)
                    // change the x axis scale
                    .y_grid_spacer(log_grid_spacer(20))
                    // add axis labels
                    .x_axis_label(localization.translate("metrics-overview-time"))
                    // format x axis values
                    .label_formatter(|_name, point| {
                        let display = format_x_value(point.x);
                        format!("x: {}\ny: {:.1}", display, point.y)
                    })
                    .x_axis_formatter(|mark, _range| format_x_value(mark.value))
                    // show the plot
                    .show(ui, |plot_ui| {
                        // display the lines
                        for line in lines {
                            plot_ui.line(line);
                        }
                    });

                // return to game
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("back"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    match stats.get() {
                        StatisticsViewState::Hidden => unreachable!(),
                        // set the pause state instead of the statsviewstate
                        // because otherwise the pausescreen would flash for a split second
                        StatisticsViewState::FromGame => pause_next.set(GamePauseState::Running),
                        // we cannot unpause the game as we want to return to the pause screen
                        StatisticsViewState::FromPaused => {
                            stats_next.set(StatisticsViewState::Hidden);
                            pauseui.set(PauseUiState::Shown);
                        }
                    }
                    click_event.send(UiClickEvent);
                }
                reset_plot.inner
            })
            .inner
        });

    if reset_plot.inner {
        if let Some(mut mem) = PlotMemory::load(egui, PLOT_ID.into()) {
            mem.auto_bounds.x = true;
            mem.auto_bounds.y = true;
            mem.set_transform(PlotTransform::new(
                *mem.transform().frame(),
                PlotBounds::NOTHING,
                Vec2b::FALSE,
            ));
            mem.store(egui, PLOT_ID.into());
        }
    }
}

/// format the input seconds as h m s string
/// the hour/minute/seconds values will only be shown if not zero
/// if nothing is shown, 0s will be displayed
/// if the input number is negative an empty string will be returned
fn format_x_value(x: f64) -> String {
    if x < 0.0 {
        // don't show label if x value is negative
        return String::new();
    }
    // calculate raw time
    let secs = x;
    let min = secs / 60.;
    let h = min / 60.;
    let mut display = String::new();
    let h = h as usize;
    if h != 0 {
        display.push_str(&format!("{}h", h));
    }
    let min = (min as usize) % 60;
    if min != 0 {
        if h != 0 {
            display.push(' ');
        }
        display.push_str(&format!("{}m", min));
    }
    let secs = (secs as usize) % 60;
    if secs != 0 || (h == 0 && min == 0) {
        if h != 0 || min != 0 {
            display.push(' ');
        }
        display.push_str(&format!("{}s", secs));
    }
    display
}

#[derive(SystemParam)]
struct StatsToggleStates<'w> {
    stats_next: ResMut<'w, NextState<StatisticsViewState>>,
    pause_next: ResMut<'w, NextState<GamePauseState>>,
    pauseui_next: ResMut<'w, NextState<PauseUiState>>,
}

/// shows/hides the statistic screen
/// should return to the pause screen if opened from the pause screen
fn toggle_statistics_view(
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    pause: Res<State<GamePauseState>>,
    stats: Option<Res<State<StatisticsViewState>>>,
    mut states: StatsToggleStates,
    mut kb_event: EventWriter<KeyboardEvent>,
    mut contexts: EguiContexts,
) {
    if is_typing_egui(&mut contexts) {
        return;
    }
    if stats.is_some()
        && settings
            .keybindings
            .menu
            .any(|code| input.just_pressed(code))
    {
        // use pause keybinding to close the stats view
        states.pause_next.set(GamePauseState::Running);
    }
    if settings
        .keybindings
        .view_stats
        .any(|code| input.just_pressed(code))
    {
        if let Some(stats) = stats {
            match stats.get() {
                StatisticsViewState::Hidden => {
                    states.pause_next.set(GamePauseState::Paused);
                    match pause.get() {
                        GamePauseState::Paused => {
                            states.stats_next.set(StatisticsViewState::FromPaused)
                        }
                        GamePauseState::Running => {
                            states.stats_next.set(StatisticsViewState::FromGame)
                        }
                    }
                    // ensure the pause screen isn't visible
                    states.pauseui_next.set(PauseUiState::Hidden);
                }
                // set the pause state instead of the statsviewstate
                // because otherwise the pausescreen would flash for a split second
                StatisticsViewState::FromGame => states.pause_next.set(GamePauseState::Running),
                // we cannot unpause the game as we want to return to the pause screen
                StatisticsViewState::FromPaused => {
                    states.stats_next.set(StatisticsViewState::Hidden);
                    states.pauseui_next.set(PauseUiState::Shown);
                }
            }
        } else {
            // if the StatisticsViewState doesn't exist,
            // the game has to be paused
            states.pause_next.set(GamePauseState::Paused);
            states.stats_next.set(StatisticsViewState::FromGame);
        }
        kb_event.send(KeyboardEvent);
    }
}
