use bevy::prelude::*;
use bevy_egui::{egui, EguiContexts};

use super::{layout::LayoutManager, IconButtonCard};

use crate::{
    define_asset_collection,
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
        },
        GamePauseState,
    },
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        in_game::metrics::ui_open_stats_button,
        is_typing_egui,
        theme::{BIG_TEXT_SIZE, OUTER_MARGIN},
        UiClickEvent,
    },
};

macro_rules! define_game_speed_plugin {
    ($({
        // state in which game_speed should be setable
        state: $state:expr,
    }),*) => {
        pub struct GameSpeedPlugin;
        impl Plugin for GameSpeedPlugin{
            fn build(&self, app: &mut App) {
                use crate::prelude::never;
                app.register_asset_collection::<SpeedIcons>();
                app.add_systems(
                    Update,
                    ui_speed_button
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(ShowGameUi::On))
                        .run_if(in_state(GamePauseState::Running))
                        .after(ui_open_stats_button),
                );

            }
        }
    }
}

define_game_speed_plugin![
    {
        state: ChallengeGameState::InRound,
    },
    {
        state: CampaignChallengeState::InRound,
    },
    {
        state: CampaignWettbewerbState::InRound,
    },
    {
        state: ZenGameState::InRound,
    }
];

define_asset_collection!(
    SpeedIcons,
    !speedx1 : Image = "icons/buttons/speedx1.png",
    !speedx2 : Image = "icons/buttons/speedx2.png",
    !speedx4 : Image = "icons/buttons/speedx4.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

#[allow(clippy::too_many_arguments)]
pub fn ui_speed_button(
    windows: Query<&Window>,
    mut layout: ResMut<LayoutManager>,
    mut contexts: EguiContexts,
    mut click_event: EventWriter<UiClickEvent>,
    icon: Res<SpeedIcons>,
    mut time: ResMut<Time<Virtual>>,
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.width() == 0.0 || window.height() == 0.0 {
        return;
    }

    let icon_size = window.height() * BIG_TEXT_SIZE / 4.0;

    let icon_texture = contexts.add_image(match time.relative_speed() {
        x if x < 1.5 => icon.speedx1.clone_weak(),
        x if (1.5..3.0).contains(&x) => icon.speedx2.clone_weak(),
        _ => icon.speedx4.clone_weak(),
    });

    let icon = IconButtonCard {
        icon_size,
        texture: icon_texture,
    };

    let cell = if let Some(cell) = layout.allocate_margin(
        super::layout::LayoutOrigin::RightTop,
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutSizeRequest::Fixed(icon.get_size()),
        super::layout::LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) {
        cell
    } else {
        return;
    };

    let mut switch_fastforward = !is_typing_egui(&mut contexts)
        && settings
            .keybindings
            .fastforward
            .any(|code| input.just_pressed(code));

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    egui::Area::new("game-speed-button".into())
        .pivot(egui::Align2::RIGHT_TOP)
        .fixed_pos(cell.rect.right_top())
        .show(egui, |ui| {
            if ui.add(icon).clicked() {
                click_event.send(UiClickEvent);
                switch_fastforward = true;
            }
        });

    if switch_fastforward {
        let mut speed = time.relative_speed() * 2.0;
        if speed > 4.0 {
            speed = 1.0;
        }
        time.set_relative_speed(speed);
    }
}
