use super::layout::{
    init_layout_manager, LayoutDirection, LayoutManager, LayoutOrigin, LayoutSizeRequest,
};
use super::{
    creative_inventory::CreativeInventoryUiState, inventory::InventorySlot, CommonInventoryAssets,
};
use crate::ui::in_game::hand::render_player_hand;
use crate::{
    game::{
        controls::CreativeInventory, gamemodes::challenge::ChallengeGameState,
        tiles::assets::TilePreviews, GamePauseState,
    },
    i18n::Localization,
    prelude::*,
};
use bevy::prelude::*;
use bevy_egui::{
    egui::{self},
    EguiContexts,
};

use super::super::{
    commons::ShowGameUi,
    theme::{INV_SLOT_SIZE, INV_SLOT_STROKE, OUTER_MARGIN},
};

macro_rules! define_debug_tile_placer_ui_plugin {
    ($({
        state: $state:expr$(,)?
    }),*) => {
        pub struct DebugTilePlacerUiPlugin;
        impl Plugin for DebugTilePlacerUiPlugin {
            fn build(&self, app: &mut App) {
                use crate::prelude::never;

                app.add_systems(
                    Update,
                    debug_placer_slots
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(in_state(ShowGameUi::On))
                        .before(render_player_hand)
                        .after(init_layout_manager)
                );
            }
        }
    }
}
define_debug_tile_placer_ui_plugin![
    {
        state: ChallengeGameState::InRound
    }
];

#[allow(clippy::too_many_arguments)]
pub fn debug_placer_slots(
    mut contexts: EguiContexts,
    mut inv: ResMut<CreativeInventory>,
    mut layout_mgr: ResMut<LayoutManager>,
    previews: Res<TilePreviews>,
    localization: Res<Localization>,
    handles: Res<CommonInventoryAssets>,
    mut inv_next: ResMut<NextState<CreativeInventoryUiState>>,
    mut pause_next: ResMut<NextState<GamePauseState>>,
) {
    let egui_open_ico = contexts.add_image(handles.menu_open.clone_weak());

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let screen_size = egui.input(|i| i.screen_rect);
    let slot_size = screen_size.width().min(screen_size.height()) * INV_SLOT_SIZE;

    let Some(cell) = layout_mgr.allocate_margin(
        LayoutOrigin::LeftBottom,
        LayoutSizeRequest::Fixed(slot_size * 2.0),
        LayoutSizeRequest::Fixed(slot_size),
        LayoutDirection::Horizontal,
        OUTER_MARGIN,
    ) else {
        return;
    };

    egui::Area::new("debug-placer-slots".into())
        .pivot(egui::Align2::CENTER_BOTTOM)
        .fixed_pos(cell.rect.center_bottom())
        .show(egui, |ui| {
            ui.horizontal(|ui| {
                // render item slots themselves
                for (index, slot) in inv.slots.clone().iter().enumerate() {
                    if ui
                        .add(InventorySlot {
                            id: (index + 1).to_string(),
                            show_id: true,
                            selected: index == inv.selected,
                            size: slot_size,
                            item: slot.map(|tile| {
                                (
                                    previews
                                        .get_egui_handle(tile)
                                        .expect("Every tile has a preview"),
                                    localization.translate(tile.get_card_translation_id()),
                                )
                            }),
                            ..Default::default()
                        })
                        .clicked()
                    {
                        inv.selected = index;
                    }
                }
                // render open inventory button
                if ui
                    .add(InventorySlot {
                        id: "inventory-open-button".to_string(),
                        show_id: false,
                        selected: false,
                        size: slot_size,
                        item: Some((egui_open_ico, localization.translate("inventory-open"))),
                        tint: INV_SLOT_STROKE,
                    })
                    .clicked()
                {
                    pause_next.set(GamePauseState::Paused);
                    inv_next.set(CreativeInventoryUiState::Opened);
                }
            });
        });
}
