// campaign mode related UI elements

use bevy::{ecs::system::SystemParam, prelude::*};
use bevy_egui::{egui, EguiContexts};

use crate::{
    game::{
        controls::KeyboardEvent,
        gamemodes::{
            campaign::{
                CampaignLevelProgress, CampaignLevelStart, CampaignModeConfig,
                UpdateCampaignStorySegment, LEVEL_TRANSLATION_ID_BASE,
            },
            GameMode,
        },
        GamePauseState,
    },
    i18n::Localization,
    settings::ActiveSettingsBank,
    ui::{
        is_using_egui,
        theme::{
            BACKGROUND_ALPHA, BETWEENROUNDS_BACKGROUND_COLOR, BUTTON_COLOR, BUTTON_TEXT_COLOR,
            GRID_TEXT_COLOR, OVERLAY_TITLE_TEXT_COLOR, PARAGRAPH_FONT_NAME, POPUP_COLOR_BG,
            POPUP_RADIUS, POPUP_STROKE_COLOR, PRIMARY_BUTTON_SIZE, PRIMARY_BUTTON_TEXT_SIZE,
            SMALL_TEXT_SIZE, SPACE_SIZE,
        },
        UiClickEvent,
    },
    AppState,
};

use super::{in_state_any, Cast, Translate};

pub struct CampaignUiPlugin;
impl Plugin for CampaignUiPlugin {
    fn build(&self, app: &mut App) {
        app.init_resource::<SegmentTextDisplay>();
        app.add_systems(
            Update,
            update_segment_text.run_if(in_state(GameMode::Campaign)),
        );
        app.add_systems(
            Update,
            display_story_text
                .run_if(in_state(CampaignLevelProgress::Running))
                .run_if(in_state(GamePauseState::Running)),
        );
        app.add_systems(
            Update,
            (skip_story_binding, display_story_binding)
                .chain()
                .run_if(in_state_any(vec![
                    CampaignLevelProgress::Intro,
                    CampaignLevelProgress::Outro,
                ])),
        );
    }
}

/// segment state storage used by the ui plugin only
/// updated using the update segment event
/// whenever new text is set, `shown` is set to `true`
#[derive(Resource, Default)]
struct SegmentTextDisplay {
    /// the translation id of the text belonging to the current section
    text: String,
    /// weather or not the info dialog is currently shown
    shown: bool,
}

/// updates the segment text display with the latest translation id
/// whenever a segment update occurs
/// and sets the shown state to true
fn update_segment_text(
    mut ev: EventReader<UpdateCampaignStorySegment>,
    mut segment_data: ResMut<SegmentTextDisplay>,
) {
    if ev.is_empty() {
        return;
    }
    for ev in ev.read() {
        segment_data.text.clone_from(&ev.0);
        segment_data.shown = true;
    }
}

#[derive(SystemParam)]
struct CampaignData<'w> {
    level_progress: Res<'w, State<CampaignLevelProgress>>,
    campaign_level_start: EventWriter<'w, CampaignLevelStart>,
    config: Res<'w, CampaignModeConfig>,
    segment_data: ResMut<'w, SegmentTextDisplay>,
}

#[derive(SystemParam)]
struct SkipStoryController<'w> {
    touches: Res<'w, Touches>,
    input: Res<'w, ButtonInput<KeyCode>>,
    mouse_input: Res<'w, ButtonInput<MouseButton>>,
    campaign_level_start: EventWriter<'w, CampaignLevelStart>,
    level_progress: Res<'w, State<CampaignLevelProgress>>,
}

/// step over intro/outro segments
/// may not be used if the level progress is Running
/// Detects mouse and touch input outside of egui elements
/// Also detects keybindings
fn skip_story_binding(
    mut contexts: EguiContexts,
    settings: Res<ActiveSettingsBank>,
    mut app_state: ResMut<NextState<AppState>>,
    mut ctrl: SkipStoryController,
    mut kb_event: EventWriter<KeyboardEvent>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    let mut should_skip = false;

    // only process touch/mouse input, when they are not interacting with egui
    if !is_using_egui(&mut contexts) {
        // skip over the section if the player clicks anywhere on the screen
        // with the mouse or on a touch screen (except for egui element interactions)
        if ctrl.mouse_input.just_released(MouseButton::Left) || ctrl.touches.any_just_released() {
            should_skip = true;
            click_event.send(UiClickEvent);
        }
    }

    // skip over section using keyboard
    // uses the same keybinding as next_round, as the screens look similar
    // and using the same keybinding reduces confusion
    if settings
        .0
        .keybindings
        .next_round
        .any(|code| ctrl.input.just_pressed(code))
    {
        should_skip = true;
        kb_event.send(KeyboardEvent);
    }

    if !should_skip {
        return;
    }

    match *ctrl.level_progress.get() {
        // show start level button
        CampaignLevelProgress::Intro => {
            ctrl.campaign_level_start.send(CampaignLevelStart);
        }
        CampaignLevelProgress::Outro => {
            app_state.set(AppState::AppMenu);
        }
        _ => unreachable!(),
    }
}

/// displays the level intro or outro text
fn display_story_binding(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut app_state: ResMut<NextState<AppState>>,
    localization: Res<Localization>,
    mut data: CampaignData,
    mut click_event: EventWriter<UiClickEvent>,
) {
    // draw background overlay
    let frame = egui::Frame {
        fill: BETWEENROUNDS_BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(contexts.ctx_mut(), |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let big_text_size = window.y / 10.0;
    let localization = localization.into_inner();

    egui::Area::new("story-bind".into())
        .anchor(egui::Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(contexts.ctx_mut(), |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                let task_id = data.segment_data.text.clone();

                let title_id = match *data.level_progress.get() {
                    CampaignLevelProgress::Intro => {
                        // get level id and construct translation id using it
                        let group = data.config.level.get_group_id();
                        let level = data.config.level.get_level_id();
                        &format!("{}-{}-{}", LEVEL_TRANSLATION_ID_BASE, group, level)
                    }
                    CampaignLevelProgress::Outro => "campaign-level_complete",
                    _ => unreachable!(),
                };

                ui.label(
                    egui::RichText::new(localization.translate(title_id))
                        .font(egui::FontId::new(
                            big_text_size,
                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                        ))
                        .color(OVERLAY_TITLE_TEXT_COLOR),
                );
                ui.label(
                    egui::RichText::new(localization.translate(&task_id))
                        .font(egui::FontId::new(
                            window.y * SMALL_TEXT_SIZE,
                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                        ))
                        .color(GRID_TEXT_COLOR),
                );

                ui.add_space(SPACE_SIZE * window.y * 3.0);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                match *data.level_progress.get() {
                    // show start level button
                    CampaignLevelProgress::Intro => {
                        if ui
                            .add_sized(
                                (window * PRIMARY_BUTTON_SIZE).cast(),
                                egui::Button::new(
                                    egui::RichText::new(localization.translate("campaign-begin"))
                                        .color(BUTTON_TEXT_COLOR),
                                )
                                .fill(BUTTON_COLOR),
                            )
                            .clicked()
                        {
                            data.campaign_level_start.send(CampaignLevelStart);
                            click_event.send(UiClickEvent);
                        }
                    }
                    // show return to menu button
                    // NOTE: when we implement a proper campaign level & group selection screen
                    // this button should probably go there instead
                    CampaignLevelProgress::Outro => {
                        if ui
                            .add_sized(
                                (window * PRIMARY_BUTTON_SIZE).cast(),
                                egui::Button::new(
                                    egui::RichText::new(
                                        localization.translate("campaign-return_to_main"),
                                    )
                                    .color(BUTTON_TEXT_COLOR),
                                )
                                .fill(BUTTON_COLOR),
                            )
                            .clicked()
                        {
                            app_state.set(AppState::AppMenu);
                            click_event.send(UiClickEvent);
                        }
                    }
                    _ => unreachable!(),
                }
            });
        });
}

/// shows a window with the segment storyline text
/// should only be used if the level is running
fn display_story_text(
    mut contexts: EguiContexts,
    mut segment_data: ResMut<SegmentTextDisplay>,
    localization: Res<Localization>,
    level_progress: Res<State<CampaignLevelProgress>>,
) {
    let my_frame = egui::Frame {
        fill: POPUP_COLOR_BG,
        stroke: egui::Stroke::new(1.5, POPUP_STROKE_COLOR),
        corner_radius: POPUP_RADIUS,
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        ..Default::default()
    };

    let task_id = segment_data.text.clone();

    let title_id = match *level_progress.get() {
        CampaignLevelProgress::Running => "campaign-task",
        _ => unreachable!(),
    };

    egui::Window::new(localization.translate(title_id))
        .frame(my_frame)
        // TODO: consider allowing the dialog to collapse instead of closing it in the future
        // might need a "new message" animation though
        .collapsible(false)
        .order(egui::Order::Tooltip)
        .open(&mut segment_data.shown)
        .id("campaign-level-segment-text".into())
        .show(contexts.ctx_mut(), |ui| {
            // show the task text
            ui.label(localization.translate(&task_id));
        });
}
