use crate::{
    define_asset_collection, i18n::Localization, prelude::*, settings::ActiveSettingsBank,
    AppMenuState,
};
use bevy::prelude::*;
use bevy_egui::{
    egui::{self, load::SizedTexture, OpenUrl},
    EguiContexts,
};

use super::{theme::*, WEBSITE_BASE};

pub struct HelpAppPlugin;
impl Plugin for HelpAppPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<HelpIcon>();
        app.add_systems(
            Update,
            open_help_link.run_if(in_state(AppMenuState::MainMenu)),
        );
    }
}

define_asset_collection!(
    HelpIcon,
    !open : Image = "icons/buttons/help.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

fn open_help_link(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    icon: Res<HelpIcon>,
    localization: Res<Localization>,
    settings: Res<ActiveSettingsBank>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    // add the icon to the egui context
    let icon_texture = contexts.add_image(icon.open.clone_weak());
    let icon_size = (window.y * BIG_TEXT_SIZE / 3.0).min(512.);
    let margin = window.y * SPACE_SIZE;

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    egui::Area::new("help-icon-button".into())
        // offset the area so that the buttons are centered
        .anchor(
            egui::Align2::RIGHT_BOTTOM,
            // x offset: margin and icon size of about button
            // and half of the margin, to indicate that the icon buttons belong together
            [-margin - icon_size - margin / 2.0, -margin],
        )
        .show(egui, |ui| {
            let resp = ui.add(
                egui::widgets::Image::new(SizedTexture::new(
                    icon_texture,
                    egui::Vec2::new(icon_size, icon_size),
                ))
                .sense(egui::Sense::click())
                .tint(BUTTON_COLOR),
            );

            if resp.clicked() {
                ui.ctx().open_url(OpenUrl::new_tab(format!(
                    "{}/{}/book/",
                    WEBSITE_BASE,
                    settings.lang.get_book_id()
                )));
            } else if resp.hovered() {
                if let Some(hover_pos) = ui.ctx().input(|i| i.pointer.hover_pos()) {
                    // manually check if the cursor hover position is inside the card rectangle
                    // because the area blocks hovering in the overlaid area
                    if resp.rect.contains(hover_pos) {
                        egui::show_tooltip_at_pointer(
                            ui.ctx(),
                            ui.layer_id(),
                            "menu-show_help-tooltip".into(),
                            |ui| {
                                ui.label(localization.translate("show-help"));
                            },
                        );
                    }
                }
            }
        });
}
