use bevy::prelude::*;
use bevy_egui::{
    egui::{self, Align2, FontId, RichText, Vec2},
    EguiContexts,
};

use crate::errors::ErrorDisplay;

use super::theme::{ERROR_BACKGROUND_COLOR, SPACE_SIZE, TITLE_FONT_NAME, TITLE_TEXT_COLOR};

pub fn show_error_screen(
    errors: Query<&ErrorDisplay>,
    mut contexts: EguiContexts,
    windows: Query<&Window>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // draw background overlay
    let frame = egui::Frame {
        fill: ERROR_BACKGROUND_COLOR,
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(egui, |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());
    let big_text_size = window.y / 10.0;

    egui::Area::new("error-screen".into())
        .anchor(Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(egui, |ui| {
            egui::ScrollArea::vertical()
                .max_width(window.x - SPACE_SIZE * window.x * 2.0)
                .max_height(window.y - SPACE_SIZE * window.y * 2.0)
                .id_salt("error-screen-content")
                .show(ui, |ui| {
                    for ErrorDisplay {
                        title,
                        description,
                        link,
                    } in &errors
                    {
                        ui.label(
                            RichText::new(title)
                                .font(FontId::new(
                                    big_text_size / 2.0,
                                    egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                ))
                                .color(TITLE_TEXT_COLOR),
                        );
                        ui.label(description);
                        if let Some(text) = link {
                            if ui.link(text).clicked() {
                                // TODO: implement link clicking behaviour
                            };
                        }
                    }
                });
        });
}
