use std::time::Duration;

use bevy::{
    diagnostic::{DiagnosticsStore, FrameTimeDiagnosticsPlugin},
    prelude::*,
    render::view::screenshot::{save_to_disk, Screenshot},
};
use bevy_egui::{
    egui::{self, Align2, FontId},
    EguiContexts,
};

use super::{
    main_menu::ui_main_menu_system,
    settings_menu::{setting_menu_system, KeyRemap},
    theme::PARAGRAPH_FONT_NAME,
};
use crate::{
    prelude::on_timer_real,
    settings::{get_screenshot_path, ActiveSettingsBank},
    AppState,
};
use chrono::{prelude::Local, DateTime};

#[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
use bevy::window::{Window, WindowMode};

pub struct CommonsPlugin;
impl Plugin for CommonsPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.init_state::<ShowFPS>();
        app.init_state::<ShowGameUi>();
        app.add_systems(
            Update,
            toggle_fps.run_if(not(in_state(KeyRemap::Listening))),
        );
        app.insert_resource(FPSData::default());
        app.add_systems(
            Update,
            (
                update_fps.run_if(on_timer_real(Duration::from_millis(100))),
                render_fps
                    .before(ui_main_menu_system)
                    .before(setting_menu_system),
            )
                .chain()
                .run_if(in_state(ShowFPS::On)),
        );
        app.add_systems(Update, toggle_fullscreen);
        app.add_systems(
            Update,
            take_screen_shot.run_if(not(in_state(KeyRemap::Listening))),
        );
        app.add_systems(Update, toggle_ui.run_if(in_state(AppState::InGame)));
        app.add_systems(OnEnter(AppState::InGame), set_ui_visible);
        app.add_event::<ScreenshotEvent>();
    }
}

#[derive(Event)]
pub struct ScreenshotEvent;

fn take_screen_shot(
    mut commands: Commands,
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
    mut event: EventWriter<ScreenshotEvent>,
) {
    if settings
        .0
        .keybindings
        .take_screenshot
        .any(|code| input.just_pressed(code))
    {
        let local: DateTime<Local> = Local::now();
        if let Some(path) = get_screenshot_path(local.format("tte-%Y-%m-%d_%H%M%S.png").to_string())
        {
            commands
                .spawn(Screenshot::primary_window())
                .observe(save_to_disk(path));

            event.send(ScreenshotEvent);
        }
    }
}

#[derive(Debug, States, Default, Hash, PartialEq, Eq, Clone)]
pub enum ShowGameUi {
    Off,
    #[default]
    On,
}

fn set_ui_visible(mut next_state: ResMut<NextState<ShowGameUi>>) {
    next_state.set(ShowGameUi::On);
}

fn toggle_ui(
    input: Res<ButtonInput<KeyCode>>,
    state: Res<State<ShowGameUi>>,
    mut next_state: ResMut<NextState<ShowGameUi>>,
    settings: Res<ActiveSettingsBank>,
) {
    if settings
        .0
        .keybindings
        .toggle_ui
        .any(|code| input.just_pressed(code))
    {
        match state.get() {
            ShowGameUi::Off => next_state.set(ShowGameUi::On),
            ShowGameUi::On => next_state.set(ShowGameUi::Off),
        }
    }
}

fn toggle_fullscreen(
    input: Res<ButtonInput<KeyCode>>,
    #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))] mut windows: Query<
        &mut Window,
    >,
    settings: Res<ActiveSettingsBank>,
) {
    #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
    /// call fullscreen on web (using „soft unwrap“)
    fn web_fullscreen() -> Option<()> {
        let window = web_sys::window()?;
        let document = window.document()?;
        // Check is fullscreen functionality is avtivated.
        if !document.fullscreen_enabled() {
            return Some(());
        }
        // Exit or Enter fullscreen
        if document.fullscreen_element().is_some() {
            document.exit_fullscreen();
        } else {
            let body: web_sys::HtmlElement = document.body()?;
            let _ = body.request_fullscreen();
        }
        Some(())
    }
    if settings
        .0
        .keybindings
        .fullscreen
        .any(|code| input.just_pressed(code))
    {
        #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
        web_fullscreen();

        #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
        // not web fullscreen
        {
            let Ok(mut window) = windows.get_single_mut() else {
                return;
            };
            if window.mode == WindowMode::Windowed {
                window.mode = WindowMode::Fullscreen(MonitorSelection::Current);
            } else {
                window.mode = WindowMode::Windowed;
            }
        }
    }
}

#[derive(Debug, States, Default, Hash, PartialEq, Eq, Clone)]
enum ShowFPS {
    #[default]
    Off,
    On,
}

#[derive(Resource)]
struct FPSData {
    fps: f64,
    had_error: bool,
}

impl Default for FPSData {
    fn default() -> Self {
        Self {
            fps: 0.0,
            had_error: false,
        }
    }
}

fn toggle_fps(
    state: Res<State<ShowFPS>>,
    mut next_state: ResMut<NextState<ShowFPS>>,
    input: Res<ButtonInput<KeyCode>>,
    settings: Res<ActiveSettingsBank>,
) {
    if settings
        .0
        .keybindings
        .toggle_fps
        .any(|code| input.just_pressed(code))
    {
        match state.get() {
            ShowFPS::Off => next_state.set(ShowFPS::On),
            ShowFPS::On => next_state.set(ShowFPS::Off),
        }
    }
}

fn update_fps(diagnostics: Res<DiagnosticsStore>, mut fps_data: ResMut<FPSData>) {
    let Some(fps) = diagnostics
        .get(&FrameTimeDiagnosticsPlugin::FPS)
        .and_then(|fps| fps.smoothed())
    else {
        if !fps_data.had_error {
            fps_data.had_error = true;
            warn!("Could not use FrameTimeDiagnosticsPlugin::FPS");
        }
        return;
    };
    fps_data.fps = fps;
}

fn render_fps(mut contexts: EguiContexts, fps_data: Res<FPSData>) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    egui::Area::new("fps".into())
        .interactable(false)
        .anchor(Align2::RIGHT_TOP, egui::Vec2::new(-5.0, 5.0))
        .order(egui::Order::Foreground)
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;
            ui.add(
                egui::Label::new(
                    egui::RichText::new(format!("fps: {:>1.0}", fps_data.fps))
                        .font(FontId::new(
                            16.0,
                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                        ))
                        .strong()
                        .color(egui::Color32::from_gray(0))
                        .background_color(egui::Color32::from_white_alpha(100)),
                )
                .extend(),
            );
        });
}
