use crate::{
    define_asset_collection, i18n::Localization, prelude::*, ui::PARAGRAPH_FONT_NAME, AppIcon,
    AppMenuState, VERSION,
};
use bevy::prelude::*;
use bevy_egui::{
    egui::{self, load::SizedTexture, FontId, Frame, RichText, ScrollArea, Ui},
    EguiContexts,
};

use super::{theme::*, UiClickEvent};

const LICENSE_URL: &str =
    "https://codeberg.org/terratactician-expandoria/game/src/branch/main/LICENSE";
const CODEBERG_URL: &str = "https://codeberg.org/terratactician-expandoria/game";

pub struct AboutAppPlugin;
impl Plugin for AboutAppPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<AboutIcon>();
        app.add_systems(
            Update,
            ui_open_about.run_if(in_state(AppMenuState::MainMenu)),
        );
        app.add_systems(
            Update,
            ui_about_system.run_if(in_state(AppMenuState::About)),
        );
    }
}

define_asset_collection!(
    AboutIcon,
    !open : Image = "icons/buttons/about.png",
    err : "buttton-icon-missing" "Button icon missing." "button-icon-missing-desc" "You can not use a couple of buttons without the icons"
);

fn ui_open_about(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    icon: Res<AboutIcon>,
    mut app_menu_state: ResMut<NextState<AppMenuState>>,
    localization: Res<Localization>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    // add the icon to the egui context
    let icon_texture = contexts.add_image(icon.open.clone_weak());
    let icon_size = (window.y * BIG_TEXT_SIZE / 3.0).min(512.);
    let margin = window.y * SPACE_SIZE;

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    egui::Area::new("about-icon-button".into())
        // offset the area so that the buttons are centered
        .anchor(egui::Align2::RIGHT_BOTTOM, [-margin, -margin])
        .show(egui, |ui| {
            let resp = ui.add(
                egui::widgets::Image::new(SizedTexture::new(
                    icon_texture,
                    egui::Vec2::new(icon_size, icon_size),
                ))
                .sense(egui::Sense::click())
                .tint(BUTTON_COLOR),
            );

            if resp.clicked() {
                app_menu_state.set(AppMenuState::About);
            } else if resp.hovered() {
                if let Some(hover_pos) = ui.ctx().input(|i| i.pointer.hover_pos()) {
                    // manually check if the cursor hover position is inside the card rectangle
                    // because the area blocks hovering in the overlaid area
                    if resp.rect.contains(hover_pos) {
                        egui::show_tooltip_at_pointer(
                            ui.ctx(),
                            ui.layer_id(),
                            "menu-about-tooltip".into(),
                            |ui| {
                                ui.label(localization.translate("about"));
                            },
                        );
                    }
                }
            }
        });
}

pub fn ui_about_system(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut app_menu_state: ResMut<NextState<AppMenuState>>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
    icon: Res<AppIcon>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    // add the app icon to the egui context
    // icon.icon will be None at this point if it failed to load
    let icon_texture = icon
        .icon
        .as_ref()
        .map(|handle| contexts.add_image(handle.clone_weak()));

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.linear_multiply(BACKGROUND_ALPHA),
        inner_margin: egui::Margin::same((window.y * MARGIN_SIZE) as i8),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    let title_text = RichText::new("TerraTactician")
        .font(egui::FontId::new(
            window.y * HEADING_SIZE,
            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
        ))
        .color(TITLE_TEXT_COLOR);
    let subtitle_text = RichText::new("- Expandoria -")
        .font(egui::FontId::new(
            window.y * SMALL_HEADING_SIZE,
            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
        ))
        .color(TITLE_TEXT_COLOR);

    egui::CentralPanel::default()
        // fill the background with the background color
        .frame(my_frame)
        .show(egui, |_ui| {});

    egui::Area::new("about-content-area".into())
        // offset the area so that the buttons are centered
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .default_width(window.x)
        .show(egui, |ui| {
            ui.style_mut().text_styles.insert(
                egui::TextStyle::Button,
                egui::FontId::new(
                    window.y * SECONDARY_BUTTON_TEXT_SIZE,
                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                ),
            );

            // max height for the info, all except: three spaces (one added, two for bottom and top padding), and one button
            let info_height = window.y * (1.0 - (SPACE_SIZE * 2.0 * 3.0 + SECONDARY_BUTTON_SIZE.y));

            ui.vertical_centered(|ui| {
                ui.style_mut().visuals.override_text_color = Some(NORMAL_TEXT_COLOR);
                dynamic_columns(
                    ui,
                    info_height,
                    |ui| {
                        // left hand / top section
                        // should contain: icon, name, version
                        // maybe: website button
                        ui.style_mut().interaction.selectable_labels = false;
                        ui.style_mut().wrap_mode = Some(egui::TextWrapMode::Extend);
                        ui.vertical_centered(|ui| {
                            // show the app icon if the icon was loaded
                            // otherwise nothing will be shown
                            // icon.icon will be None at this point if it failed to load
                            if let Some(texture_id) = icon_texture {
                                let screen_size = ui.ctx().input(|i| i.screen_rect);
                                let size = screen_size.height()
                                    * PRIMARY_BUTTON_SIZE.y.min(APP_ICON_RESOLUTION);
                                ui.image(SizedTexture::new(
                                    texture_id,
                                    egui::Vec2::new(size, size),
                                ));
                                ui.add_space(window.y * SPACE_SIZE);
                            }

                            ui.label(title_text);
                            ui.label(subtitle_text);
                            ui.label(format!("v{VERSION}"));
                        });
                    },
                    |ui| {
                        // right hand /bottom section
                        // should contain: developers, license information
                        // and more links if needed
                        {
                            // core developers section
                            ui.label(
                                RichText::new(localization.translate("team"))
                                    .font(FontId::new(
                                        window.y * SMALL_HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ))
                                    .color(TITLE_TEXT_COLOR),
                            );
                            for name in [
                                "Benedikt Marschner",
                                "Calvin Klein",
                                "Jakob Meier",
                                "Julian Sievers",
                                "Nils Lange",
                            ] {
                                list_entry(ui, |ui| ui.label(name));
                            }
                        }
                        ui.add_space(window.y * SPACE_SIZE);
                        {
                            // source code section
                            ui.label(
                                RichText::new(localization.translate("source-code"))
                                    .font(FontId::new(
                                        window.y * SMALL_HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ))
                                    .color(TITLE_TEXT_COLOR),
                            );

                            ui.hyperlink_to(localization.translate("tte-gpl-license"), LICENSE_URL);
                            ui.hyperlink_to(
                                localization.translate("tte-codeberg-source"),
                                CODEBERG_URL,
                            );
                        }
                        ui.add_space(window.y * SPACE_SIZE);
                        {
                            // license section
                            ui.label(
                                RichText::new(localization.translate("licenses"))
                                    .font(FontId::new(
                                        window.y * SMALL_HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ))
                                    .color(TITLE_TEXT_COLOR),
                            );
                            ui.label(localization.translate("other-projects"));
                            for (name, link, license) in [
                                (
                                    "Bevy Engine",
                                    "https://github.com/bevyengine/bevy",
                                    "MIT/Apache2.0",
                                ),
                                (
                                    "Catppuccin Color palette",
                                    "https://github.com/catppuccin/catppuccin",
                                    "MIT",
                                ),
                                ("Egui", "https://github.com/emilk/egui", "MIT/Apache2.0"),
                                ("Fira", "https://github.com/mozilla/Fira", "OFL"),
                                (
                                    "Gnome Icon Development Kit",
                                    "https://gitlab.gnome.org/Teams/Design/icon-development-kit",
                                    "CC0 1.0 Universal",
                                ),
                                ("Inter", "https://rsms.me/inter/", "OFL"),
                                (
                                    "Kenney Board Game Icons",
                                    "https://kenney.nl/assets/board-game-icons",
                                    "CC0 1.0",
                                ),
                                (
                                    "Kenney Hexagon Kit",
                                    "https://kenney.nl/assets/hexagon-kit",
                                    "CC0",
                                ),
                                (
                                    "Permanent Marker",
                                    "https://fonts.google.com/specimen/Permanent+Marker",
                                    "Apache2.0",
                                ),
                            ] {
                                list_entry(ui, |ui| {
                                    ui.hyperlink_to(format!("{} ({})", name, license), link)
                                });
                            }
                        }
                        {
                            // external contributors section
                            ui.label(
                                RichText::new(localization.translate("external-contributors"))
                                    .font(FontId::new(
                                        window.y * SMALL_HEADING_SIZE,
                                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                                    ))
                                    .color(TITLE_TEXT_COLOR),
                            );
                            for name in [
                                "johnpetersa19",
                                "Colin \"cwpute\" Etienne Jean-Marie Landreau",
                            ] {
                                list_entry(ui, |ui| ui.label(name));
                            }
                        }
                    },
                );

                ui.add_space(window.y * SPACE_SIZE * 2.0);
                // back button
                if ui
                    .add_sized(
                        (window * SECONDARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            RichText::new(localization.translate("back")).color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    app_menu_state.set(AppMenuState::MainMenu);
                    click_event.send(UiClickEvent);
                }
            });
        });
}

/// draws a unordered list entry
fn list_entry<R>(ui: &mut Ui, add_contents: impl FnOnce(&mut Ui) -> R) -> R {
    // based on the egui RadioButton and column code
    let spacing = &ui.spacing();
    let icon_width = spacing.icon_width;
    let icon_spacing = spacing.icon_spacing;

    let top_left = ui.cursor().min;

    let offset = icon_width + icon_spacing * 2.0;
    let pos = top_left + egui::Vec2::new(offset, 0.0);
    let child_rect = egui::Rect::from_min_max(
        pos,
        ui.max_rect().right_bottom() - egui::Vec2::new(offset, 0.0),
    );

    let mut child_ui = ui.new_child(
        egui::UiBuilder::new()
            .max_rect(child_rect)
            .layout(egui::Layout::top_down_justified(egui::Align::LEFT)),
    );
    child_ui.set_width(child_rect.width());

    let response = add_contents(&mut child_ui);

    let (small_icon_rect, _) = ui.spacing().icon_rectangles(child_rect);

    let painter = ui.painter();
    painter.add(egui::epaint::CircleShape {
        center: top_left + egui::Vec2::new(offset / 2.0, child_ui.min_size().y / 2.0),
        radius: small_icon_rect.width() / 3.0,
        fill: NORMAL_TEXT_COLOR,
        stroke: Default::default(),
    });

    ui.advance_cursor_after_rect(egui::Rect::from_min_size(top_left, child_ui.min_size()));

    response
}

/// shorthand for `dynamic-columns_dyn`
/// with the rect set to the screen size
pub fn dynamic_columns<R>(
    ui: &mut Ui,
    max_height: f32,
    section1: impl FnOnce(&mut Ui) -> R,
    section2: impl FnOnce(&mut Ui) -> R,
) -> (R, R) {
    let screen_size = ui.ctx().input(|i| i.screen_rect);
    dynamic_columns_dyn(ui, screen_size, max_height, section1, section2)
}

/// In landscape mode (rect is wider than tall),
/// a two column layout will be used, where the second
/// column is wrapped in a scroll view
///
/// In portrait mode, both sections will be placed beneath oneanother and
/// wrapped in a single scroll view
///
/// The `SPACE_SIZE` is automatically applied dependending on the direction
pub fn dynamic_columns_dyn<R>(
    ui: &mut Ui,
    rect: egui::Rect,
    max_height: f32,
    section1: impl FnOnce(&mut Ui) -> R,
    section2: impl FnOnce(&mut Ui) -> R,
) -> (R, R) {
    let is_landscape = rect.width() > rect.height();
    if is_landscape {
        ui.style_mut().spacing.item_spacing.x = SPACE_SIZE * rect.width();
        ui.columns_const(|[left, right]| {
            // show the first column fixed on the left
            let a = left.columns_const(|[_, right]| section1(right));
            // only make the second column scrollable
            let b = ScrollArea::vertical()
                .max_height(max_height)
                .show(right, |ui| {
                    Frame::new()
                        .inner_margin(rect.width() * SPACE_SIZE)
                        .show(ui, |ui| {
                            ui.vertical_centered(|ui| ui.vertical(|ui| section2(ui)).inner)
                                .inner
                        })
                        .inner
                });
            (a, b.inner)
        })
    } else {
        ScrollArea::vertical()
            .max_height(max_height)
            .show(ui, |ui| {
                ui.vertical_centered(|ui| {
                    let a = section1(ui);
                    ui.add_space(rect.height() * SPACE_SIZE);
                    let b = ui.vertical(|ui| section2(ui));
                    (a, b.inner)
                })
                .inner
            })
            .inner
    }
}
