pub mod audio;
pub mod game_behaviour;
pub mod graphics;
pub mod keybindings;
pub mod log;
pub mod mouse;

use bevy::prelude::*;
use serde::{Deserialize, Serialize};
#[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
use std::{fs, io::Write, path};

use crate::{i18n::locales::Language, APP_ID};

use self::{
    audio::AudioSettings, game_behaviour::GameBehaviourSettings, graphics::GraphicsSettings,
    keybindings::KeybindingSettings, log::LogSettings, mouse::MouseSettings,
};

#[derive(Serialize, Deserialize, Default, Debug, Clone)]
#[serde(default)]
pub struct GameSettings {
    #[serde(default)]
    pub lang: Language,
    #[serde(default)]
    pub log: LogSettings,
    #[serde(default)]
    pub graphics: GraphicsSettings,
    #[serde(default)]
    pub mouse: MouseSettings,
    #[serde(default)]
    pub keybindings: KeybindingSettings,
    #[serde(default)]
    pub audio: AudioSettings,
    #[serde(default)]
    pub game_behaviour: GameBehaviourSettings,
}

impl GameSettings {
    #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
    pub fn from_file(path: &str) -> Result<Self, GameSettingsError> {
        if let Ok(content) = fs::read_to_string(path) {
            match toml::from_str(&content) {
                Err(e) => Err(GameSettingsError::InvalidConfig(e)),
                Ok(config) => Ok(config),
            }
        } else {
            Err(GameSettingsError::ConfigDoesNotExist)
        }
    }
    #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
    pub fn from_file(path: &str) -> Result<Self, GameSettingsError> {
        if let Some(window) = web_sys::window() {
            if let Ok(Some(local_storage)) = window.local_storage() {
                if let Ok(Some(content)) = local_storage.get_item(path) {
                    return match toml::from_str(&content) {
                        Err(e) => Err(GameSettingsError::InvalidConfig(e)),
                        Ok(config) => Ok(config),
                    };
                }
            }
        }

        Err(GameSettingsError::ConfigDoesNotExist)
    }

    #[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
    pub fn to_file(&self, path: &str) -> Result<(), GameSettingsError> {
        match toml::to_string(self) {
            Ok(buffer) => {
                // create parent directories if required
                let path = path::Path::new(path);
                if let Some(parent) = path.parent() {
                    if fs::create_dir_all(parent).is_err() {
                        return Err(GameSettingsError::UnableToWriteConfig);
                    }
                }
                // load the config file
                if let Ok(mut file) = fs::File::create(path) {
                    if file.write_all(buffer.as_bytes()).is_err() {
                        return Err(GameSettingsError::UnableToWriteConfig);
                    }
                    Ok(())
                } else {
                    Err(GameSettingsError::UnableToCreateConfig)
                }
            }
            Err(e) => Err(GameSettingsError::SerialisationFailed(e)),
        }
    }
    #[cfg(any(target_arch = "wasm32", target_arch = "wasm64"))]
    pub fn to_file(&self, path: &str) -> Result<(), GameSettingsError> {
        if let Some(window) = web_sys::window() {
            if let Ok(Some(local_storage)) = window.local_storage() {
                return match toml::to_string(self) {
                    Ok(buffer) => {
                        if local_storage.set_item(path, &buffer).is_ok() {
                            Ok(())
                        } else {
                            Err(GameSettingsError::UnableToCreateConfig)
                        }
                    }
                    Err(e) => Err(GameSettingsError::SerialisationFailed(e)),
                };
            }
        }
        Err(GameSettingsError::UnableToWriteConfig)
    }
}

#[derive(Debug)]
pub enum GameSettingsError {
    /// file not found
    ConfigDoesNotExist,
    /// toml parser error
    #[allow(dead_code)]
    InvalidConfig(toml::de::Error),
    /// serde was unable to generate a toml string
    #[allow(dead_code)]
    SerialisationFailed(toml::ser::Error),
    /// unable to open the config file for writing
    UnableToCreateConfig,
    /// couldn't write the bytes to the file
    UnableToWriteConfig,
}

pub fn get_screenshot_path(name: String) -> Option<String> {
    #[cfg(target_os = "android")]
    if let Some(Some(mut dir)) = bevy::window::ANDROID_APP
        .get()
        .map(|a| a.external_data_path())
    {
        // this should be writable without requesting permissions
        dir.push("screenshots");
        dir.push(name);
        return dir.as_path().to_str().map(|s| s.to_string());
    }

    if cfg!(any(target_arch = "wasm32", target_arch = "wasm64")) {
        return Some(name);
    }

    let base_dir = dirs::picture_dir()?;
    if let Some(path) = std::path::Path::new(&base_dir).join(name).to_str() {
        return Some(path.to_string());
    }
    None
}

pub fn get_record_name(name: String) -> Option<String> {
    #[cfg(target_os = "android")]
    if let Some(Some(mut dir)) = bevy::window::ANDROID_APP
        .get()
        .map(|a| a.external_data_path())
    {
        // this should be writable without requesting permissions
        dir.push("records");
        dir.push(name);
        return dir.as_path().to_str().map(|s| s.to_string());
    }

    if cfg!(any(target_arch = "wasm32", target_arch = "wasm64")) {
        return Some(name);
    }

    let base_dir = dirs::download_dir()?;
    if let Some(path) = std::path::Path::new(&base_dir).join(name).to_str() {
        return Some(path.to_string());
    }
    None
}

pub fn get_data_dir() -> Option<String> {
    #[cfg(target_os = "android")]
    if let Some(Some(dir)) = bevy::window::ANDROID_APP
        .get()
        .map(|a| a.external_data_path())
    {
        // this should be writable without requesting permissions
        return dir.as_path().to_str().map(|s| s.to_string());
    }

    let base_dir = dirs::data_dir()?;
    if let Some(path) = std::path::Path::new(&base_dir).join(APP_ID).to_str() {
        return Some(path.to_string());
    }

    None
}

pub fn get_config_path() -> Option<String> {
    if cfg!(any(target_arch = "wasm32", target_arch = "wasm64")) {
        return Some(String::from(APP_ID));
    }

    if let Ok(env_overwrite) = std::env::var("TTE_CONFIG") {
        return Some(env_overwrite);
    }

    let data_dir = get_data_dir()?;
    if let Some(path) = std::path::Path::new(&data_dir)
        .join("settings.toml")
        .to_str()
    {
        return Some(path.to_string());
    }
    None
}

/// disk settings version
/// loaded from disk on startup
/// stored on disk if changed
/// used to restore settings
#[derive(Resource, Deref, DerefMut)]
pub struct PersistentSettingsBank(pub GameSettings);

/// active game settings
/// can be restored from BankA
/// should update BankA onConfirm
#[derive(Resource, Deref, DerefMut)]
pub struct ActiveSettingsBank(pub GameSettings);

pub struct GameSettingsPlugin {
    pub disk_config: GameSettings,
    pub write_to_disk: bool,
}
impl Plugin for GameSettingsPlugin {
    fn build(&self, app: &mut App) {
        app.insert_resource(PersistentSettingsBank(self.disk_config.clone()));
        app.insert_resource(ActiveSettingsBank(self.disk_config.clone()));
        if self.write_to_disk {
            app.add_systems(
                Update,
                save_settings_on_change.run_if(resource_changed::<PersistentSettingsBank>),
            );
        }
    }
}

fn save_settings_on_change(settings: Res<PersistentSettingsBank>) {
    if let Some(config_file) = get_config_path() {
        match settings.to_file(&config_file) {
            Ok(()) => debug!("Updated config file"),
            Err(msg) => error!("Couldn't save config file: {:?}", msg),
        }
    }
}
