use bevy::input::keyboard::KeyCode;
use bevy::reflect::Enum;

use serde::{Deserialize, Serialize};

use crate::iter_config;

#[macro_export]
macro_rules! iter_config {
    ($name:ident, $type:ty, [$(($id:ident, $tid:literal),)*]) => {
        #[derive(Serialize, Deserialize, Debug, Clone)]
        #[serde(default)]
        pub struct $name {
            $(
                pub $id: $type,
            )*
        }
        impl $name {
            pub fn list(&self) -> Vec<String> {
                vec![
                    $($tid.to_string(),)*
                ]
            }
            pub fn get_mut(&mut self, id: &str) -> Option<&mut $type> {
                match id {
                    $(
                        $tid => Some(&mut self.$id),
                    )*
                    _ => None
                }
            }
            pub fn get(&self, id: &str) -> Option<&$type> {
                match id {
                    $(
                        $tid => Some(&self.$id),
                    )*
                    _ => None
                }
            }
        }
    };
}

impl Default for KeybindingSettings {
    fn default() -> Self {
        Self {
            toggle_fps: Keybinding::new(Some(KeyCode::F3), None),
            toggle_ui: Keybinding::new(Some(KeyCode::F1), None),
            take_screenshot: Keybinding::new(Some(KeyCode::F2), None),
            fullscreen: Keybinding::new(Some(KeyCode::F11), None),
            top_down_view: Keybinding::new(Some(KeyCode::KeyX), None),
            center_map: Keybinding::new(Some(KeyCode::KeyC), None),
            view_stats: Keybinding::new(Some(KeyCode::Tab), None),
            close_popup: Keybinding::new(Some(KeyCode::Delete), None),
            select_card_1: Keybinding::new(Some(KeyCode::Digit1), None),
            select_card_2: Keybinding::new(Some(KeyCode::Digit2), None),
            select_card_3: Keybinding::new(Some(KeyCode::Digit3), None),
            select_card_4: Keybinding::new(Some(KeyCode::Digit4), None),
            select_card_5: Keybinding::new(Some(KeyCode::Digit5), None),
            select_card_prev: Keybinding::new(Some(KeyCode::KeyQ), None),
            select_card_next: Keybinding::new(Some(KeyCode::KeyE), None),
            redraw_cards: Keybinding::new(Some(KeyCode::Space), None),
            pan_top: Keybinding::new(Some(KeyCode::KeyW), None),
            pan_left: Keybinding::new(Some(KeyCode::KeyA), None),
            pan_bot: Keybinding::new(Some(KeyCode::KeyS), None),
            pan_right: Keybinding::new(Some(KeyCode::KeyD), None),
            orbit_top: Keybinding::new(Some(KeyCode::ArrowUp), None),
            orbit_left: Keybinding::new(Some(KeyCode::ArrowLeft), None),
            orbit_bot: Keybinding::new(Some(KeyCode::ArrowDown), None),
            orbit_right: Keybinding::new(Some(KeyCode::ArrowRight), None),
            zoom_in: Keybinding::new(Some(KeyCode::NumpadAdd), None),
            zoom_out: Keybinding::new(Some(KeyCode::Minus), None),
            menu: Keybinding::new(Some(KeyCode::Escape), None),
            next_round: Keybinding::new(Some(KeyCode::Space), None),
            move_cursor_forward: Keybinding::new(Some(KeyCode::KeyI), None),
            move_cursor_forward_right: Keybinding::new(Some(KeyCode::KeyO), None),
            move_cursor_backward_right: Keybinding::new(Some(KeyCode::KeyL), None),
            move_cursor_backward: Keybinding::new(Some(KeyCode::KeyK), None),
            move_cursor_backward_left: Keybinding::new(Some(KeyCode::KeyJ), None),
            move_cursor_forward_left: Keybinding::new(Some(KeyCode::KeyU), None),
            cursor_click_left: Keybinding::new(Some(KeyCode::KeyH), None),
            cursor_click_right: Keybinding::new(Some(KeyCode::KeyY), None),
            inventory: Keybinding::new(Some(KeyCode::KeyB), None),
            inventory_focus_up: Keybinding::new(Some(KeyCode::ArrowUp), None),
            inventory_focus_down: Keybinding::new(Some(KeyCode::ArrowDown), None),
            inventory_focus_left: Keybinding::new(Some(KeyCode::ArrowLeft), None),
            inventory_focus_right: Keybinding::new(Some(KeyCode::ArrowRight), None),
            inventory_take: Keybinding::new(Some(KeyCode::KeyT), None),
            fastforward: Keybinding::new(Some(KeyCode::KeyF), None),
            menu_prev: Keybinding::new(Some(KeyCode::ArrowLeft), None),
            menu_next: Keybinding::new(Some(KeyCode::ArrowRight), None),
            menu_back: Keybinding::new(Some(KeyCode::Escape), None),
            menu_select: Keybinding::new(Some(KeyCode::Enter), None),
            scroll_swap: Keybinding::new(Some(KeyCode::ShiftLeft), Some(KeyCode::ShiftRight)),
        }
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Clone)]
pub struct Keybinding {
    pub slot1: Option<KeyCode>,
    pub slot2: Option<KeyCode>,
}
impl Keybinding {
    pub fn new(slot1: Option<KeyCode>, slot2: Option<KeyCode>) -> Self {
        Self { slot1, slot2 }
    }

    /// Tests if any element of the iterator matches a predicate.
    /// `any()` takes a closure that returns `true` or `false`. It applies
    /// this closure to each slot value that is not None, and if any of them return
    /// `true`, then so does `any()`. If they all return `false`, it
    /// returns `false`.
    ///
    /// `any()` is short-circuiting; in other words, it will stop processing
    /// as soon as it finds a `true`, given that no matter what else happens,
    /// the result will also be `true`.
    ///
    /// If all slots are empty, any will return `false`
    pub fn any<B>(&self, f: B) -> bool
    where
        B: Fn(KeyCode) -> bool,
    {
        if let Some(key) = self.slot1 {
            if f(key) {
                return true;
            }
        }
        if let Some(key) = self.slot2 {
            if f(key) {
                return true;
            }
        }
        false
    }
}

pub trait WithTranslationID {
    fn get_translation_id(&self) -> &str;
}

impl<T: WithTranslationID> WithTranslationID for Option<T> {
    fn get_translation_id(&self) -> &str {
        match self {
            Some(v) => v.get_translation_id(),
            None => "unset",
        }
    }
}

iter_config!(
    KeybindingSettings,
    Keybinding,
    [
        (toggle_fps, "config-key-toggle_fps"),
        (toggle_ui, "config-key-toggle_ui"),
        (take_screenshot, "config-key-take_screenshot"),
        (fullscreen, "config-key-fullscreen"),
        (top_down_view, "config-key-top_down_view"),
        (center_map, "config-key-center_map"),
        (view_stats, "config-key-view_stats"),
        (close_popup, "config-key-close_popup"),
        (select_card_1, "config-key-select_card_1"),
        (select_card_2, "config-key-select_card_2"),
        (select_card_3, "config-key-select_card_3"),
        (select_card_4, "config-key-select_card_4"),
        (select_card_5, "config-key-select_card_5"),
        (select_card_prev, "config-key-select_card_prev"),
        (select_card_next, "config-key-select_card_next"),
        (redraw_cards, "config-key-redraw"),
        (pan_top, "config-key-pan_top"),
        (pan_left, "config-key-pan_left"),
        (pan_bot, "config-key-pan_bot"),
        (pan_right, "config-key-pan_right"),
        (orbit_top, "config-key-orbit_top"),
        (orbit_left, "config-key-orbit_left"),
        (orbit_bot, "config-key-orbit_bot"),
        (orbit_right, "config-key-orbit_right"),
        (zoom_in, "config-key-zoom_in"),
        (zoom_out, "config-key-zoom_out"),
        (menu, "config-key-menu"),
        (next_round, "config-key-next_round"),
        (move_cursor_forward, "config-key-cursor-forward"),
        (move_cursor_forward_right, "config-key-cursor-forward_right"),
        (
            move_cursor_backward_right,
            "config-key-cursor-backward_right"
        ),
        (move_cursor_backward, "config-key-cursor-backward"),
        (move_cursor_backward_left, "config-key-cursor-backward_left"),
        (move_cursor_forward_left, "config-key-cursor-forward_left"),
        (cursor_click_left, "config-key-cursor-click_left"),
        (cursor_click_right, "config-key-cursor-click_right"),
        (inventory, "config-key-inventory"),
        (inventory_focus_up, "config-key-inventory-focus_up"),
        (inventory_focus_down, "config-key-inventory-focus_down"),
        (inventory_focus_left, "config-key-inventory-focus_left"),
        (inventory_focus_right, "config-key-inventory-focus_right"),
        (inventory_take, "config-key-inventory-take"),
        (fastforward, "config-key-fastforward"),
        (menu_select, "config-key-menu_select"),
        (menu_back, "config-key-menu_back"),
        (menu_prev, "config-key-menu_prev"),
        (menu_next, "config-key-menu_next"),
        (scroll_swap, "config-key-scroll_swap"),
    ]
);

impl WithTranslationID for KeyCode {
    fn get_translation_id(&self) -> &str {
        self.variant_name()
    }
}
