use super::args::CliArgs;
use bevy::app::App;
use std::process::ExitCode;

#[cfg(target_family = "wasm")]
pub fn setup_map(_app: &mut App, _args: &CliArgs) -> Result<bool, ExitCode> {
    Ok(false)
}

#[cfg(not(target_family = "wasm"))]
pub fn setup_map(app: &mut App, args: &CliArgs) -> Result<bool, ExitCode> {
    use bevy::prelude::*;
    use std::fs;

    #[cfg(feature = "graphics")]
    use terratactician_expandoria::DisplayMode;
    use terratactician_expandoria::{game::map::GameWorld, AppState};

    #[cfg(target_family = "wasm")]
    use terratactician_expandoria::quit;

    if let Some(path) = &args.map_file {
        if let Ok(content) = fs::read_to_string(path) {
            match serde_json::from_str::<GameWorld>(&content) {
                Ok(map) => {
                    app.add_systems(
                        OnEnter(AppState::InGame),
                        (move |mut commands: Commands,
                               #[cfg(feature = "graphics")] display_mode: Res<
                            State<DisplayMode>,
                        >| {
                            #[cfg(not(feature = "graphics"))]
                            let without_scene = true;
                            #[cfg(feature = "graphics")]
                            let without_scene = *display_mode.get() == DisplayMode::Headless;

                            map.map.build_to_world(&mut commands, without_scene);
                        })
                        .run_if(run_once),
                    );
                }

                Err(e) => {
                    #[cfg(target_family = "wasm")]
                    quit();
                    error!("Error while parsing GameWorld: {e}");
                    return Err(ExitCode::FAILURE);
                }
            }
        } else {
            #[cfg(target_family = "wasm")]
            quit();
            error!("Error while loading GameWorld: file does not exists.");
            return Err(ExitCode::FAILURE);
        }
        Ok(true)
    } else {
        Ok(false)
    }
}
