#![cfg_attr(not(debug_assertions), windows_subsystem = "windows")]

mod main_tools;

use std::process::{ExitCode, Termination};

use bevy::prelude::*;

use main_tools::{args::CliArgs, setup_game::setup_game, setup_replay};

#[cfg(target_family = "wasm")]
use terratactician_expandoria::quit;
use terratactician_expandoria::{
    game::gamemodes::{challenge::ChallengeModeConfig, zen::ZenModeConfig},
    log::{LoggingConf, LoggingController},
    AppBuilder, DisplayMode, GameConfig,
};

fn main() -> ExitCode {
    let logctr = LoggingController::init(LoggingConf::default());

    let args = CliArgs::init();

    #[cfg(feature = "graphics")]
    let display_mode = if args.display_mode.headless {
        DisplayMode::Headless
    } else {
        DisplayMode::Graphic
    };
    #[cfg(not(feature = "graphics"))]
    let display_mode = DisplayMode::Headless;

    let mut app = AppBuilder {
        display_mode,
        #[cfg(feature = "graphics")]
        enable_config_io: !args.disable_config,
        logging_controller: Some(logctr),
    }
    .build();

    #[cfg(feature = "graphics")]
    if let DisplayMode::Graphic = display_mode {
        app.add_plugins(main_tools::window_management::WindowManager);
    }

    let mut config = GameConfig::default();

    if let Some(bot) = &args.bot {
        config.use_bot = true;
        config.bot_url.clone_from(bot);
        config.skip_bot = true;
    }

    if let Some(record) = &args.recorder_file {
        info!("Game will be recorded to: {}", record);
        config.record_game = Some(record.clone());
        config.replay_name = args.replay_name.clone().unwrap_or_default();
        config.activate_recorder = true;
        config.skip_recorder = true;
    }

    config.skip_seed = args.seed.is_some();
    let seed = args
        .seed
        .clone()
        .unwrap_or_else(|| fastrand::u64(..).to_string());
    app.insert_resource(ChallengeModeConfig {
        difficulty: Default::default(),
        seed: seed.clone(),
    });
    app.insert_resource(ZenModeConfig { seed: seed.clone() });

    let mut started = false;

    // replay
    if !started {
        started = match setup_replay::setup_replay(&mut app, &args, &mut config) {
            Ok(is) => is,
            Err(err) => return err,
        };
    }

    if !started {
        // automatically configure & start game
        started = match setup_game(&mut app, &args, &mut config) {
            Ok(runns) => runns,
            Err(err) => return err,
        };
    }

    config.report_file = args.report_file;

    app.insert_resource(config);

    #[cfg(feature = "graphics")]
    let headless = args.display_mode.headless;

    #[cfg(not(feature = "graphics"))]
    let headless = true;

    if !started && headless {
        error!("No game started. Please specify a replay file or choose a game mode.");

        // XXX: don't quit the web version, as it makes it impossible to view the error message
        // because it closes the tab

        return ExitCode::FAILURE;
    }

    let app_exit = app.run();

    #[cfg(target_family = "wasm")]
    if !headless {
        // XXX: don't quit the web version, as it makes it impossible to view the log message
        // because it closes the tab
        quit();
    }

    app_exit.report()
}
